<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Product Labels compatibility with Hyva Theme
 */

namespace Amasty\ProductLabelsHyva\ViewModel;

use Amasty\Base\Model\Serializer;
use Amasty\Label\Block\Label as LabelBlock;
use Amasty\Label\Model\Source\TooltipStatus;
use Magento\Framework\View\Element\Block\ArgumentInterface;

class Label implements ArgumentInterface
{
    private const MODE = 'mode';
    private const POSITION = 'position';
    private const ALIGNMENT = 'alignment';
    private const TOOLTIP = 'tooltip';
    private const TOOLTIP_STATUS = 'status';
    private const TOOLTIP_BACKGROUND_COLOR = 'backgroundColor';
    private const TOOLTIP_BACKGROUND_DEFAULT_COLOR = '#ffffff';
    private const TOOLTIP_TEXT_COLOR = 'color';
    private const TOOLTIP_TEXT_DEFAULT_COLOR = '#000000';
    private const TOOLTIP_CONTENT = 'content';

    /**
     * @var Serializer
     */
    private Serializer $serializer;

    /**
     * @var array
     */
    private array $labelConfig;

    public function __construct(
        Serializer $serializer
    ) {
        $this->serializer = $serializer;
    }

    public function setConfigFromBlock(LabelBlock $block): void
    {
        $config = $this->serializer->unserialize($block->getJsonConfig());

        if (!is_array($config)) {
            throw new \InvalidArgumentException('Can\'t serialize label config.');
        }

        $this->labelConfig = $config;
    }

    public function getMode(): string
    {
        return $this->labelConfig[self::MODE] ?? 'cat';
    }

    public function getPosition(): string
    {
        return $this->labelConfig[self::POSITION] ?? 'top-left';
    }

    public function getAlignment(): int
    {
        return $this->labelConfig[self::ALIGNMENT] ?? 0;
    }

    public function isShowTooltip(): bool
    {
        $status = $this->getTooltip()[self::TOOLTIP_STATUS] ?? TooltipStatus::DISABLED;

        return $status !== TooltipStatus::DISABLED;
    }

    public function getTooltipBackground(): string
    {
        return $this->getTooltip()[self::TOOLTIP_BACKGROUND_COLOR] ?? self::TOOLTIP_BACKGROUND_DEFAULT_COLOR;
    }

    public function getTooltipTextColor(): string
    {
        return $this->getTooltip()[self::TOOLTIP_TEXT_COLOR] ?? self::TOOLTIP_TEXT_DEFAULT_COLOR;
    }

    public function getTooltipContent(): string
    {
        $content = $this->getTooltip()[self::TOOLTIP_CONTENT] ?? '';

        // phpcs:ignore Magento2.Functions.DiscouragedFunction.Discouraged
        return nl2br(rawurldecode($content));
    }

    private function getTooltip(): array
    {
        return $this->labelConfig[self::TOOLTIP] ?? [];
    }
}
