<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Export Products for Magento 2
 */

namespace Amasty\ProductExport\Model\Connection;

use Amasty\ProductExport\Api\ConnectionRepositoryInterface;
use Amasty\ProductExport\Api\Data\ConnectionInterface;
use Amasty\ProductExport\Api\Data\ConnectionInterfaceFactory;
use Amasty\ProductExport\Model\Connection\ResourceModel\CollectionFactory;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;

class Repository implements ConnectionRepositoryInterface
{
    /**
     * @var ConnectionInterfaceFactory
     */
    private $connectionFactory;

    /**
     * @var ResourceModel\Connection
     */
    private $connectionResource;

    /**
     * @var array
     */
    private $connections = [];

    /**
     * @var CollectionFactory
     */
    private $collectionFactory;

    public function __construct(
        ConnectionInterfaceFactory $connectionFactory,
        ResourceModel\Connection $connectionResource,
        CollectionFactory $collectionFactory
    ) {
        $this->connectionFactory = $connectionFactory;
        $this->connectionResource = $connectionResource;
        $this->collectionFactory = $collectionFactory;
    }

    public function getById(int $id): ConnectionInterface
    {
        if (!isset($this->connections[$id])) {
            /** @var ConnectionInterface $connection */
            $connection = $this->connectionFactory->create();
            $this->connectionResource->load($connection, $id);
            if (!$connection->getConnectionId()) {
                throw new NoSuchEntityException(__('Connection with specified ID "%1" not found.', $id));
            }
            $this->connections[$id] = $connection;
        }

        return $this->connections[$id];
    }

    public function save(ConnectionInterface $connection): ConnectionInterface
    {
        try {
            $resourceConnection = $this->connectionResource->getConnection();
            $referencedTableName = $this->connectionResource->getTable($connection->getTableToJoin());
            if (!$resourceConnection->isTableExists($referencedTableName)) {
                throw new CouldNotSaveException(__('Database table with specified name does not exist.'));
            }
            if (!$resourceConnection->tableColumnExists($referencedTableName, $connection->getReferencedTableKey())) {
                throw new CouldNotSaveException(
                    __('Column with specified name does not exist in the specified database table.')
                );
            }
            if ($this->getByTableAndParent(
                $connection->getTableToJoin(),
                $connection->getParentEntity()
            )->getConnectionId() && $connection->isObjectNew()
            ) {
                throw new CouldNotSaveException(
                    __('Connection with table to join %1 already exists.', $connection->getTableToJoin())
                );
            }

            $connection->setEntityCode(uniqid());
            $this->connectionResource->save($connection);

            unset($this->connections[$connection->getConnectionId()]);
        } catch (\Exception $e) {
            throw new CouldNotSaveException(__('Unable to save the connection. Error: %1', $e->getMessage()));
        }

        return $connection;
    }

    public function delete(ConnectionInterface $connection): bool
    {
        try {
            $this->connectionResource->delete($connection);
            unset($this->connections[$connection->getConnectionId()]);
        } catch (\Exception $e) {
            throw new CouldNotDeleteException(__('Unable to delete the connection. Error: %1', $e->getMessage()));
        }

        return true;
    }

    public function deleteById(int $id): bool
    {
        $connection = $this->getById($id);

        return $this->delete($connection);
    }

    public function getEmptyConnectionModel(): ConnectionInterface
    {
        return $this->connectionFactory->create();
    }

    public function getByTableAndParent(?string $tableToJoin, ?string $parentEntity): ConnectionInterface
    {
        /** @var ConnectionInterface $connection */
        $connection = $this->collectionFactory->create()
            ->addFieldToFilter(Connection::TABLE_TO_JOIN, $tableToJoin)
            ->addFieldToFilter(Connection::PARENT_ENTITY, $parentEntity)
            ->getFirstItem();

        return $connection;
    }
}
