<?php
declare(strict_types=1);
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Out Of Stock Notifications Hyva Compatibility
 */

namespace Amasty\OutofstockHyvaCompatibility\ViewModel\Catalog;

use Magento\Framework\View\Element\Block\ArgumentInterface;
use Magento\Store\Model\ScopeInterface;
use Magento\Framework\Registry;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Url\Helper\Data;
use Magento\ConfigurableProduct\Model\ResourceModel\Product\Type\Configurable;
use Magento\CatalogInventory\Api\StockRegistryInterface;
use Amasty\Xnotif\Helper\Config;
use Magento\Framework\Serialize\Serializer\Json;

/**
 * ViewModel allows pass needed configuration to template
 */
class OutofstockData implements ArgumentInterface
{
    /**
     * @var string
     */
    public const PRICE_ALERT_BLOCK = 'price';

    /**
     * @var string
     */
    public const STOCK_ALERT_BLOCK = 'stock';

    /**
     * @var Registry
     */
    private $registry;

    /**
     * @var ScopeConfigInterface
     */
    private $scopeConfig;

    /**
     * @var Data
     */
    private $helper;

    /**
     * @var Configurable
     */
    private $configurable;

    /**
     * @var StockRegistryInterface
     */
    private $stockRegistry;

    /**
     * @var Config
     */
    private $config;

    /**
     * @var Json
     */
    private $json;

    /**
     * @param Registry $registry
     * @param ScopeConfigInterface $scopeConfig
     * @param Data $helper
     * @param Configurable $configurable
     * @param StockRegistryInterface $stockRegistry
     * @param Config $config
     * @param Json $json
     */
    public function __construct(
        Registry                $registry,
        ScopeConfigInterface    $scopeConfig,
        Data                    $helper,
        Configurable            $configurable,
        StockRegistryInterface  $stockRegistry,
        Config                  $config,
        Json                    $json
    ) {
        $this->registry = $registry;
        $this->scopeConfig = $scopeConfig;
        $this->helper = $helper;
        $this->configurable = $configurable;
        $this->stockRegistry = $stockRegistry;
        $this->config = $config;
        $this->json = $json;
    }

    /**
     * Check if customer is logged in
     *
     * @return bool
     */
    public function isCustomerLoggedIn(): bool
    {
        return $this->config->isLoggedIn();
    }

    /**
     * Get product
     *
     * @return mixed
     */
    public function getProduct()
    {
        return $this->registry->registry('product');
    }

    /**
     * Add stock availability for configurable simple products
     *
     * @param array $productId
     * @return array
     */
    public function getAllChildren($productId): array
    {
        $childProducts = $this->configurable->getChildrenIds($productId);
        $childProducts = array_keys($childProducts[0]);
        $isSaleable = [];

        foreach ($childProducts as $product) {
            $isSaleable[$product] = $this->stockRegistry->getStockItem($product)['is_in_stock'];
        }

        return $isSaleable;
    }

    /**
     * Get original product
     *
     * @param mixed $block
     * @return mixed
     */
    public function getOriginalProduct($block)
    {
        $product = $block->getOriginalProduct();
        if (!$product) {
            $product = $this->getProduct();
        }

        return $product;
    }

    /**
     * Check if gdpr agreement config is enabled in Admin
     *
     * @param string|int|null $store
     * @return mixed
     */
    public function isGDRPEnabled($store = null): string
    {
        return $this->scopeConfig->getValue(
            'amxnotif/gdrp/enabled',
            ScopeInterface::SCOPE_STORE,
            $store
        );
    }

    /**
     * Get gdpr label config in Admin
     *
     * @param string|int|null $store
     * @return string
     */
    public function getGDRPText($store = null): string
    {
        return (string)$this->scopeConfig->getValue(
            'amxnotif/gdrp/text',
            ScopeInterface::SCOPE_STORE,
            $store
        );
    }

    /**
     * Get stock placeholder config in Admin
     *
     * @param int|null|string $store
     * @return string
     */
    public function getPlaceholder($store = null): string
    {
        return (string)$this->scopeConfig->getValue(
            'amxnotif/stock/placeholder',
            ScopeInterface::SCOPE_STORE,
            $store
        );
    }

    /**
     * Get configuration for showing options of Out Of Stock Configurable
     *
     * @param int|null|string $store
     * @return string
     */
    public function isOutofstockOfOutofstockEnabled($store = null): string
    {
        return (string)$this->scopeConfig->getValue(
            'amxnotif/general/show_out_of_stock_only',
            ScopeInterface::SCOPE_STORE,
            $store
        );
    }

    /**
     * Get configuration for show alert price
     *
     * @param int|null|string $store
     * @return string
     */
    public function isAllowAlertPrice($store = null): string
    {
        return (string)$this->scopeConfig->getValue(
            'catalog/productalert/allow_price',
            ScopeInterface::SCOPE_STORE,
            $store
        );
    }

    /**
     * Get configuration for show alert stock
     *
     * @param int|null|string $store
     * @return string
     */
    public function isAllowAlertStock($store = null): string
    {
        return (string)$this->scopeConfig->getValue(
            'catalog/productalert/allow_stock',
            ScopeInterface::SCOPE_STORE,
            $store
        );
    }

    /**
     * Encode url for hidden input
     *
     * @param string|null $url
     * @return string
     */
    public function encodeUrl(?string $url = null): string
    {
        return $this->helper->getEncodedUrl($url);
    }

    /**
     * Check if alert block can be shown for current customer group
     *
     * @param string $alertBlockType
     *
     * @return bool
     */
    public function isAllowedForCurrentCustomerGroup(string $alertBlockType): bool
    {
        return $this->config->allowForCurrentCustomerGroup($alertBlockType);
    }

    /**
     * Encode array to string
     *
     * @param array $data
     *
     * @return string
     */
    public function serializeArray(array $data): string
    {
        try {
            return $this->json->serialize($data);
        } catch (\InvalidArgumentException $e) {
            return '';
        }
    }
}
