<?php

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Out Of Stock Notifications Hyva Compatibility
 */

declare(strict_types=1);

namespace Amasty\OutofstockHyvaCompatibility\Plugins\Catalog\Block\Product;

use Amasty\Xnotif\Block\Catalog\Category\StockSubscribe;
use Amasty\Xnotif\Plugins\Catalog\Block\Product\AbstractProduct as Base;
use Magento\Catalog\Block\Product\AbstractProduct as ProductBlock;
use Hyva\Theme\Service\CurrentTheme;
use Amasty\Xnotif\Helper\Data;
use Amasty\Xnotif\Helper\Config;
use Magento\Catalog\Model\Product as ProductModel;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable as ConfigurableModel;
use Magento\Framework\Registry;
use Magento\Framework\View\Element\BlockInterface;
use Magento\Framework\View\LayoutInterface;
use Magento\Framework\Exception\LocalizedException;

class AbstractProduct extends Base
{
    /**
     * @var string
     */
    public const STOCK_STATUS_BLOCK_NAME = 'category.info.stockstatus';

    /**
     * @var CurrentTheme
     */
    private $currentTheme;

    /**
     * @var Registry
     */
    private $registry;

    /**
     * @var Data
     */
    private $xnotifHelper;

    /**
     * @var array
     */
    private $processedProducts = [];

    /**
     * @var string
     */
    private $stockStatusTemplate;

    /**
     * @var string
     */
    private $loggedTemplate;

    /**
     * @var string
     */
    private $notLoggedTemplate;

    /**
     * @param Data $xnotifHelper
     * @param Config $config
     * @param Registry $registry
     * @param CurrentTheme $currentTheme
     * @param string $stockStatusTemplate
     * @param string $loggedTemplate
     * @param string $notLoggedTemplate
     */
    public function __construct(
        Data $xnotifHelper,
        Config $config,
        Registry $registry,
        CurrentTheme $currentTheme,
        string $stockStatusTemplate = 'Amasty_OutofstockHyvaCompatibility::category/stock-status.phtml',
        string $loggedTemplate = 'Magento_ProductAlert::product/view.phtml',
        string $notLoggedTemplate = 'Amasty_Xnotif::category/subscribe.phtml'
    ) {
        parent::__construct($xnotifHelper, $config, $registry);
        $this->currentTheme = $currentTheme;
        $this->registry = $registry;
        $this->xnotifHelper = $xnotifHelper;
        $this->stockStatusTemplate = $stockStatusTemplate;
        $this->loggedTemplate = $loggedTemplate;
        $this->notLoggedTemplate = $notLoggedTemplate;
    }

    /**
     * Retrieve review summary html
     *
     * @param ProductBlock $subject
     * @param string $result
     *
     * @return string
     */
    public function afterGetReviewsSummaryHtml(ProductBlock $subject, $result): string
    {
        $output = parent::afterGetReviewsSummaryHtml($subject, $result);

        if (!$this->registry->registry('current_product') && $this->currentTheme->isHyva()) {
            $output .= $this->getStockStatusHtml($subject);
        }

        return $output;
    }

    /**
     * Get subscribe block to output
     *
     * @param ProductBlock $subject
     * @return string
     */
    public function getSubscribeHtml($subject): string
    {
        $html = '';
        $layout = $subject->getLayout();
        $product = $this->getProduct();

        if (!$this->currentTheme->isHyva()) {
            if (!$product->isSaleable() && !$product->getData('amxnotif_hide_alert')) {
                $html = $this->generateAlertHtml($layout, $product);
            }

            if ($product->getTypeId() == ConfigurableModel::TYPE_CODE) {
                $html .= $this->generateConfigurableHtml();
            }
        } else {
            if (($product->getTypeId() == ConfigurableModel::TYPE_CODE
                || ($product->getTypeId() != ConfigurableModel::TYPE_CODE && !$product->getIsSalable()))
                && !$product->getData('amxnotif_hide_alert')
            ) {
                $html = $this->generateAlertHtml($layout, $product);
            }
        }

        $this->processedProducts[$subject->getNameInLayout()][] = $product->getId();
        $this->registry->unregister('xnotifConfigurables');
        $this->registry->register('xnotifConfigurables', $this->processedProducts[$subject->getNameInLayout()]);

        return $html;
    }

    /**
     * Prepare subscribe block to output
     *
     * @param BlockInterface $subscribeBlock
     * @param ProductModel $product
     * @param string $refererUrl
     * @return void
     */
    protected function prepareSubscribeBlock(BlockInterface $subscribeBlock, $product, $refererUrl = null)
    {
        if ($this->xnotifHelper->isLoggedIn()) {
            if (!$this->currentTheme->isHyva()) {
                parent::prepareSubscribeBlock($subscribeBlock, $product, $refererUrl);
            }
        } else {
            $subscribeBlock->setTemplate($this->notLoggedTemplate);
            $subscribeBlock->setOriginalProduct($product);
        }
    }

    /**
     * Retrieve stock status html
     *
     * @param ProductBlock $productBlock
     *
     * @return string
     */
    private function getStockStatusHtml(ProductBlock $productBlock): string
    {
        try {
            $layout = $productBlock->getLayout();
            $product = $this->getProduct();
            $html = $this->generateStockStatusHtml($layout, $product);

            $this->processedProducts[$productBlock->getNameInLayout()][] = $product->getId();
            $this->registry->unregister('xnotifConfigurables');
            $this->registry->register(
                'xnotifConfigurables',
                $this->processedProducts[$productBlock->getNameInLayout()]
            );
        } catch (LocalizedException $e) {
            $html = '';
        }

        return $html;
    }

    /**
     * Generate stock status block to output
     *
     * @param LayoutInterface $layout
     * @param ProductModel $product
     *
     * @return string
     */
    private function generateStockStatusHtml(LayoutInterface $layout, ProductModel $product): string
    {
        $stockStatusBlock = $this->getStockStatusBlock($layout);
        $currentProduct = $this->registry->registry('current_product');

        /*check if it is child product for replace product registered to child product.*/
        $isChildProduct = $currentProduct && ($currentProduct->getId() != $product->getId());
        if ($isChildProduct) {
            $stockStatusBlock->setData('parent_product_id', $currentProduct->getId());
            $stockStatusBlock->setOriginalProduct($product);
        }

        $stockStatusBlock->setTemplate($this->stockStatusTemplate);
        $stockStatusBlock->setOriginalProduct($product);

        return $stockStatusBlock->setProduct($product)->toHtml();
    }

    /**
     * Get stock status block to output
     *
     * @param LayoutInterface $layout
     *
     * @return BlockInterface
     */
    private function getStockStatusBlock(LayoutInterface $layout): BlockInterface
    {
        $subscribeBlock = $layout->getBlock(self::STOCK_STATUS_BLOCK_NAME);
        if (!$subscribeBlock) {
            $subscribeBlock = $layout->createBlock(StockSubscribe::class, self::STOCK_STATUS_BLOCK_NAME);
        }

        return $subscribeBlock;
    }

    /**
     * Retrieve container html
     *
     * @return string
     */
    private function generateConfigurableHtml(): string
    {
        $productId = $this->getProduct()->getId();

        return sprintf('<div class="amxnotif-category-container" data-amsubscribe="%s"></div>', $productId);
    }
}
