<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Out of stock Notifications GraphQL for Magento 2 (System)
 */

namespace Amasty\OsnGraphql\Test\GraphQl;

use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Framework\GraphQl\Query\Uid;
use Magento\Integration\Api\CustomerTokenServiceInterface;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\GraphQlAbstract;

class AmxnotifStockSubscribeTest extends GraphQlAbstract
{
    public const CUSTOMER_NAME = 'customer@example.com';
    public const CUSTOMER_PASS = 'password';
    public const MAIN_QUERY_KEY = 'AmxnotifStockSubscribe';
    public const RESPONSE_KEY = 'response_message';

    /**
     * @var CustomerTokenServiceInterface
     */
    private $customerTokenService;

    /**
     * @var Uid
     */
    private $idEncoder;

    /**
     * @var ProductRepositoryInterface
     */
    private $productRepository;

    protected function setUp(): void
    {
        parent::setUp();

        $objectManager = Bootstrap::getObjectManager();
        $this->customerTokenService = $objectManager->get(CustomerTokenServiceInterface::class);
        $this->idEncoder = $objectManager->create(Uid::class);
        $this->productRepository = $objectManager->get(ProductRepositoryInterface::class);
    }

    /**
     * @group amasty_osn
     *
     * @magentoApiDataFixture Magento/Customer/_files/customer.php
     * @magentoApiDataFixture Magento/GraphQl/Catalog/_files/simple_product.php
     */
    public function testAmxnotifStockSubscribe()
    {
        $product = $this->productRepository->get('simple_product');
        $productId = $product->getId();
        $productUid = $this->idEncoder->encode((string)$productId);

        $variables = [
            'email' => self::CUSTOMER_NAME,
            'gdprAgreement' => true,
            'productId' => $productId,
            'productUid' => $productUid
        ];
        $query = $this->getQuery();
        $response = $this->graphQlMutation($query, $variables, '', $this->getHeader());
        $responseSecondTime = $this->graphQlMutation($query, $variables, '', $this->getHeader());

        $this->assertArrayHasKey(self::MAIN_QUERY_KEY, $response);
        $this->assertArrayHasKey(self::RESPONSE_KEY, $response[self::MAIN_QUERY_KEY]);
        $this->assertEquals(
            'Alert subscription has been saved.',
            $response[self::MAIN_QUERY_KEY][ self::RESPONSE_KEY]
        );

        $this->assertArrayHasKey(self::MAIN_QUERY_KEY, $responseSecondTime);
        $this->assertArrayHasKey(self::RESPONSE_KEY, $responseSecondTime[self::MAIN_QUERY_KEY]);
        $this->assertEquals(
            'Thank you! You are already subscribed to this product.',
            $responseSecondTime[self::MAIN_QUERY_KEY][ self::RESPONSE_KEY]
        );
    }

    /**
     * @return string
     */
    private function getQuery(): string
    {
        return <<<'MUTATION'
mutation AmxnotifStock(
  $email: String,
  $gdprAgreement: Boolean,
  $productId: Int,
  $productUid: String
){
  AmxnotifStockSubscribe(
    input: {
      email:$email
      gdpr_agreement:$gdprAgreement
      product_id:$productId
      product_uid:$productUid
    }
  ) {
     response_message
  }
}
MUTATION;
    }

    /**
     * @param string $username
     * @param string $password
     *
     * @return string[]
     */
    private function getHeader(
        string $username = self::CUSTOMER_NAME,
        string $password = self::CUSTOMER_PASS
    ): array {
        $customerToken = $this->customerTokenService->createCustomerAccessToken($username, $password);
        return ['Authorization' => 'Bearer ' . $customerToken];
    }
}
