<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Order Attributes GraphQl for Magento 2
 */

namespace Amasty\OrderattrGraphql\Model\Resolver;

use Amasty\Orderattr\Api\CheckoutDataRepositoryInterface;
use Amasty\Orderattr\Model\Entity\Adapter\Quote\Adapter;
use Amasty\Orderattr\Model\Entity\EntityDataFactory;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Framework\GraphQl\Query\Resolver\ContextInterface;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Framework\Message\MessageInterface;
use Magento\QuoteGraphQl\Model\Cart\GetCartForUser;

class SaveAttributes implements ResolverInterface
{
    private const CART_ID_KEY = 'cartId';
    private const ORDER_ATTRIBUTES_KEY = 'orderAttributes';

    /**
     * @var CheckoutDataRepositoryInterface
     */
    private $dataRepository;

    /**
     * @var EntityDataFactory
     */
    private $entityDataFactory;

    /**
     * @var GetCartForUser
     */
    private $getCartForUser;

    /**
     * @var Adapter
     */
    private $quoteAdapter;

    public function __construct(
        GetCartForUser $getCartForUser,
        CheckoutDataRepositoryInterface $dataRepository,
        Adapter $quoteAdapter,
        EntityDataFactory $entityDataFactory
    ) {
        $this->getCartForUser = $getCartForUser;
        $this->dataRepository = $dataRepository;
        $this->quoteAdapter = $quoteAdapter;
        $this->entityDataFactory = $entityDataFactory;
    }

    /**
     * @param Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return array
     * @throws GraphQlInputException
     */
    public function resolve(Field $field, $context, ResolveInfo $info, array $value = null, array $args = null)
    {
        if (!isset($args[self::CART_ID_KEY]) || empty($args[self::CART_ID_KEY])) {
            throw new GraphQlInputException(__('Required parameter "%1" is missing', self::CART_ID_KEY));
        }
        if (!isset($args[self::ORDER_ATTRIBUTES_KEY]) || empty($args[self::ORDER_ATTRIBUTES_KEY])) {
            throw new GraphQlInputException(__('Required parameter "%1" is missing', self::ORDER_ATTRIBUTES_KEY));
        }

        $allErrors = [];

        $currentUserId = $context->getUserId();
        $storeId = (int)$context->getExtensionAttributes()->getStore()->getId();
        $cart = $this->getCartForUser->execute($args[self::CART_ID_KEY], $currentUserId, $storeId);

        $entityData = $this->entityDataFactory->create();

        foreach ($args[self::ORDER_ATTRIBUTES_KEY] as $attributeData) {
            $entityData->setCustomAttribute($attributeData['attribute_code'], $attributeData['value']);
        }

        $shippingMethod = '';
        if ($cart->getShippingAddress()) {
            $shippingMethod = $cart->getShippingAddress()->getShippingMethod();
        }

        try {
            $this->dataRepository->save($cart->getEntityId(), 'amasty_checkout', $shippingMethod, $entityData);
            $this->quoteAdapter->addExtensionAttributesToQuote($cart, true);
        } catch (\Exception $e) {
            $allErrors[] = $e->getMessage();
        }

        if ($cart->getData('has_error')) {
            $errors = $cart->getErrors();

            /** @var MessageInterface $error */
            foreach ($errors as $error) {
                $allErrors[] = $error->getText();
            }
        }

        return [
            'cart' => [
                'model' => $cart,
            ],
            'user_errors' => $allErrors
        ];
    }
}
