<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Order Attributes GraphQl for Magento 2
 */

namespace Amasty\OrderattrGraphql\Model\Resolver;

use Amasty\Orderattr\Model\Attribute\RelationValidator;
use Amasty\Orderattr\Model\Entity\EntityData;
use Amasty\Orderattr\Model\Entity\EntityResolver;
use Amasty\Orderattr\Model\Value\Metadata\Form;
use Amasty\Orderattr\Model\Value\Metadata\FormFactory;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Framework\GraphQl\Query\Resolver\ContextInterface;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;

class GetSavedAttributesData implements ResolverInterface
{
    /**
     * @var FormFactory
     */
    private $metadataFormFactory;

    /**
     * @var EntityResolver
     */
    private $entityResolver;

    /**
     * @var RelationValidator
     */
    private $relationValidator;

    public function __construct(
        FormFactory $metadataFormFactory,
        EntityResolver $entityResolver,
        RelationValidator $relationValidator
    ) {
        $this->metadataFormFactory = $metadataFormFactory;
        $this->entityResolver = $entityResolver;
        $this->relationValidator = $relationValidator;
    }

    /**
     * @param Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return array
     * @throws GraphQlInputException
     */
    public function resolve(Field $field, $context, ResolveInfo $info, array $value = null, array $args = null)
    {
        if (!isset($value['model'])) {
            throw new GraphQlInputException(__('"model" value must be specified'));
        }

        $modelEntity = $value['model'];
        $result = [];
        $shippingMethod = '';

        if ($modelEntity instanceof \Magento\Sales\Api\Data\OrderInterface) {
            $entity = $this->entityResolver->getEntityByOrder($modelEntity);
            $shippingMethod = $modelEntity->getShippingMethod();
        } else {
            $entity = $this->entityResolver->getEntityByQuoteId($modelEntity->getId());
            if ($modelEntity->getShippingAddress()) {
                $shippingMethod = $modelEntity->getShippingAddress()->getShippingMethod() ?? '';
            }
        }

        $form = $this->createEntityForm(
            $entity,
            (int)$modelEntity->getStore()->getId(),
            (int)$modelEntity->getCustomerGroupId(),
            $shippingMethod
        );
        $outputData = $form->outputData(Form::FORMAT_TO_VALIDATE_RELATIONS);

        // array: dependent_attribute_code => bool
        // if true - value should be shown
        $attributesToShow = $this->relationValidator->getAttributesToShow($outputData, $entity);

        foreach ($outputData as $attributeCode => $attributeValue) {
            if (!array_key_exists($attributeCode, $attributesToShow) || $attributesToShow[$attributeCode]) {
                if (is_array($attributeValue)) {
                    $attributeValue = current($attributeValue);
                }
                $result[] = [
                        'attribute_code' => $attributeCode,
                        'value' => $attributeValue
                    ];
            }
        }

        return $result;
    }

    protected function createEntityForm(
        EntityData $entity,
        int $storeId,
        int $customerGroupId,
        string $shippingMethod
    ): Form {
        /** @var Form $formProcessor */
        $formProcessor = $this->metadataFormFactory->create();
        $formProcessor->setFormCode('graphql_view')
            ->setEntity($entity)
            ->setShippingMethod($shippingMethod)
            ->setStore($storeId)
            ->setCustomerGroupId($customerGroupId);

        return $formProcessor;
    }
}
