<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Order Attributes GraphQl for Magento 2
 */

namespace Amasty\OrderattrGraphql\Model\Resolver;

use Amasty\Orderattr\Model\Attribute\Attribute;
use Amasty\Orderattr\Model\Config\Source\CheckoutStep;
use Amasty\Orderattr\Model\QuoteProducts;
use Amasty\Orderattr\Model\ResourceModel\Attribute\CollectionFactory;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Framework\GraphQl\Query\Resolver\ContextInterface;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\QuoteGraphQl\Model\Cart\GetCartForUser;

class AvailableAttributes implements ResolverInterface
{
    public const ATTRIBUTES_WITH_OPTIONS = [
        'checkboxes',
        'radios',
        'boolean', // Yes/No attribute type
        'multiselect',
        'select'
    ];
    private const CART_ID_KEY = 'cartId';

    /**
     * @var GetCartForUser
     */
    private $getCartForUser;

    /**
     * @var CollectionFactory
     */
    private $collectionFactory;

    /**
     * @var QuoteProducts
     */
    private $quoteProducts;

    public function __construct(
        GetCartForUser $getCartForUser,
        CollectionFactory $collectionFactory,
        QuoteProducts $quoteProducts
    ) {
        $this->getCartForUser = $getCartForUser;
        $this->collectionFactory = $collectionFactory;
        $this->quoteProducts = $quoteProducts;
    }

    /**
     * @param Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return array
     * @throws GraphQlInputException
     */
    public function resolve(
        Field $field,
        $context,
        ResolveInfo $info,
        array $value = null,
        array $args = null
    ) {
        if (!isset($args[self::CART_ID_KEY]) || empty($args[self::CART_ID_KEY])) {
            throw new GraphQlInputException(__('Required parameter "%1" is missing', self::CART_ID_KEY));
        }

        $currentUserId = $context->getUserId();
        $storeId = (int)$context->getExtensionAttributes()->getStore()->getId();
        $cart = $this->getCartForUser->execute($args[self::CART_ID_KEY], $currentUserId, $storeId);
        $shippingMethod = '';
        if ($cart->getShippingAddress()) {
            $shippingMethod = $cart->getShippingAddress()->getShippingMethod();
        }

        $attributeCollection = $this->collectionFactory->create();
        $attributeCollection->addCustomerGroupFilter((int)$cart->getCustomerGroupId())
            ->addShippingMethodsFilter($shippingMethod)
            ->addStoreFilter($storeId)
            ->addConditionsFilter($this->quoteProducts->getProductIds($cart));

        $result = [];
        foreach ($attributeCollection->getItems() as $attribute) {
            if (!$attribute->getIsHiddenFromCustomer()
                && $attribute->getIsVisibleOnFront()
                && ((int)$attribute->getCheckoutStep() !== CheckoutStep::NONE)
            ) {
                $attributeData = $attribute->getData();
                $attributeData['options'] = $this->getAttributeOptions($attribute);
                $result[] = $attributeData;
            }
        }

        return $result;
    }

    private function getAttributeOptions(Attribute $attribute): array
    {
        if (in_array($attribute->getFrontendInput(), self::ATTRIBUTES_WITH_OPTIONS, true)) {
            return $attribute->getSource()->getAllOptions(false);
        }

        return [];
    }
}
