<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
 * @package Page Speed Optimizer Premium Functionality for Magento 2 (System)
 */

namespace Amasty\OptimizerPremiumFunctionality\Observer;

use Amasty\ImageOptimizer\Model\Image\ImageSettingGetter;
use Amasty\ImageOptimizer\Model\ImageProcessor;
use Amasty\ImageOptimizer\Model\Queue\ImageQueueService;
use Amasty\OptimizerPremiumFunctionality\Model\Constants;
use Amasty\OptimizerPremiumFunctionality\Model\ImageOptimizerConfigProvider as ConfigProvider;
use Magento\Catalog\Model\Product\Media\Config;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;

class ProcessProductImages implements ObserverInterface
{
    /**
     * @var ImageProcessor
     */
    private $imageProcessor;

    /**
     * @var Config
     */
    private $mediaConfig;

    /**
     * @var ImageQueueService
     */
    private $imageQueueService;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var ImageSettingGetter
     */
    private $imageSettingGetter;

    public function __construct(
        ImageProcessor $imageProcessor,
        ImageQueueService $imageQueueService,
        Config $mediaConfig,
        ConfigProvider $configProvider,
        ImageSettingGetter $imageSettingGetter
    ) {
        $this->imageProcessor = $imageProcessor;
        $this->mediaConfig = $mediaConfig;
        $this->imageQueueService = $imageQueueService;
        $this->configProvider = $configProvider;
        $this->imageSettingGetter = $imageSettingGetter;
    }

    public function execute(Observer $observer)
    {
        if (!$this->configProvider->isOptimizeImagesByCron()) {
            return;
        }
        $product = $observer->getProduct();
        if ($images = $product->getMediaGallery('images')) {
            foreach ($images as $image) {
                $realPath = $this->mediaConfig->getMediaPath($image['file']);
                $imageSetting = $this->imageSettingGetter->getSettingByImgPath($realPath);
                if (!isset($imageSetting) || !$imageSetting->isAutomaticOptimization()) {
                    continue;
                }
                if (isset($image['removed']) && $image['removed'] == '1') {
                    $this->imageQueueService->deleteByFilename($realPath);
                }
                if (!isset($image['file'], $image['new_file'])) {
                    continue;
                }
                $queueImage = $this->imageProcessor->prepareQueue($realPath, $imageSetting);
                if ($queueImage) {
                    $queueImage->setQueueType(Constants::CRON_QUEUE);
                    $this->imageQueueService->addToQueue($queueImage);
                }
            }
        }
    }
}
