<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
 * @package Page Speed Optimizer Premium Functionality for Magento 2 (System)
 */

namespace Amasty\OptimizerPremiumFunctionality\Model\ImageProcessor\AutoProcessing;

use Amasty\ImageOptimizer\Model\Image\ImageSettingGetter;
use Amasty\ImageOptimizer\Model\ImageProcessor;
use Amasty\ImageOptimizer\Model\ImageProcessor\AutoProcessing\Processors\AutoProcessorInterface;
use Amasty\ImageOptimizer\Model\Queue\ImageQueueService;
use Amasty\OptimizerPremiumFunctionality\Model\Constants;
use Amasty\OptimizerPremiumFunctionality\Model\ImageOptimizerConfigProvider as ConfigProvider;

class CronProcessImage implements AutoProcessorInterface
{
    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var ImageProcessor
     */
    private $imageProcessor;

    /**
     * @var ImageQueueService
     */
    private $imageQueueService;

    /**
     * @var array
     */
    private $invalidPaths;

    /**
     * @var ImageSettingGetter
     */
    private $imageSettingGetter;

    public function __construct(
        ImageProcessor     $imageProcessor,
        ConfigProvider     $configProvider,
        ImageQueueService  $imageQueueService,
        ImageSettingGetter $imageSettingGetter,
        array              $invalidPaths = []
    ) {
        $this->configProvider = $configProvider;
        $this->imageProcessor = $imageProcessor;
        $this->imageQueueService = $imageQueueService;
        $this->imageSettingGetter = $imageSettingGetter;
        $this->invalidPaths = $invalidPaths;
    }

    public function execute(string $imgPath): void
    {
        if (!$this->configProvider->isOptimizeImagesByCron()
            || !$this->isPathValid($imgPath)
            || ($imageSetting = $this->imageSettingGetter->getSettingByImgPath($imgPath)) === null
            || !$imageSetting->isAutomaticOptimization()
        ) {
            return;
        }

        if ($queueImage = $this->imageProcessor->prepareQueue($imgPath, $imageSetting)) {
            $queueImage->setQueueType(Constants::CRON_QUEUE);
            $this->imageQueueService->addToQueue($queueImage);
        }
    }

    private function isPathValid(string $imgPath): bool
    {
        foreach ($this->invalidPaths as $invalidPath) {
            if (strpos($imgPath, $invalidPath) !== false) {
                return false;
            }
        }

        return true;
    }
}
