<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Amasty_MetaTagsToElasticSearch
*/


declare(strict_types=1);

namespace Amasty\MetaTagsToElasticSearch\Model\Indexer;

use Amasty\Meta\Helper\Data as MetaHelper;
use Magento\Catalog\Model\ProductFactory;
use Magento\Framework\App\Area;
use Magento\Framework\App\State;
use Magento\Store\Model\App\Emulation;

class ReplaceDataProvider
{
    /**
     * @var MetaHelper
     */
    private $metaHelper;

    /**
     * @var ProductFactory
     */
    private $productFactory;

    /**
     * @var Emulation
     */
    private $appEmulation;

    /**
     * @var State
     */
    private $appState;

    /**
     * @var array
     */
    private $attributeMapping = [
        'meta_keywords' => 'meta_keyword'
    ];

    public function __construct(
        MetaHelper $metaHelper,
        ProductFactory $productFactory,
        Emulation $appEmulation,
        State $appState
    ) {
        $this->metaHelper = $metaHelper;
        $this->productFactory = $productFactory;
        $this->appEmulation = $appEmulation;
        $this->appState = $appState;
    }

    /**
     * Method provides data for replacement
     *
     * @param array $documentData
     * @param int $storeId
     * @return array
     * @throws \Exception
     */
    public function getDataForReplace(array $documentData, int $storeId): array
    {
        $this->appEmulation->startEnvironmentEmulation($storeId);
        $dataForReplace = $this->appState->emulateAreaCode(
            Area::AREA_FRONTEND,
            [$this, 'getProductDataForMap'],
            [$documentData]
        );
        $this->appEmulation->stopEnvironmentEmulation();

        return $dataForReplace;
    }

    /**
     * Callback method for area code emulator
     *
     * @param array $productsData
     * @return array
     */
    public function getProductDataForMap(array $productsData): array
    {
        $result = [];
        $product = $this->productFactory->create();
        foreach ($productsData as $productId => $productData) {
            $product->setEntityId($productId);
            foreach ($productData as $attributeCode => $attributeValue) {
                if ($attributeValue !== null && is_string($attributeValue)) {
                    $product->setData($attributeCode, $attributeValue);
                }
            }
            $result[$productId] = $this->mapAttributes(
                $this->metaHelper->observeProductPage($product) ?: []
            );
            $product->unsetData();
        }

        return $result;
    }

    private function mapAttributes(array $productData): array
    {
        foreach ($productData as $attributeCode => $attributeValue) {
            if (isset($this->attributeMapping[$attributeCode])) {
                $productData[$this->attributeMapping[$attributeCode]] = $attributeValue;
                unset($productData[$attributeCode]);
            }
        }

        return $productData;
    }
}
