<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Google Invisible Captcha Hyva
 */

namespace Amasty\InvisibleCaptchaHyva\Model\GraphQl;

use Amasty\InvisibleCaptcha\Model\Captcha;
use Amasty\InvisibleCaptcha\Model\ConfigProvider;
use Amasty\InvisibleCaptchaHyva\Api\GraphQlCaptchaValidatorInterface;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\App\Request\Http as HttpRequest;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Framework\Message\ManagerInterface as MessageManager;

/**
 * Validate reCaptcha on graphQl queries
 *
 * For custom forms validation, add mutation name to the "Request URL of the Protected Form" config
 * and add 'X-ReCaptcha' header to the mutation
 */
class GraphQlCaptchaValidator implements GraphQlCaptchaValidatorInterface
{
    public const CAPTCHA_HEADER_PARAM = 'X-ReCaptcha';

    /**
     * @var HttpRequest
     */
    private HttpRequest $request;

    /**
     * @var Captcha
     */
    private Captcha $captchaModel;

    /**
     * @var ConfigProvider
     */
    private ConfigProvider $configProvider;

    /**
     * If Default Magento Forms have appropriate graphql endpoint it should be stored in this variable.
     * Key - Mutation name, value - default form action
     *
     * @var string[]
     */
    private array $graphQlDefaultFormsRelation;

    /**
     * @var MessageManager
     */
    private MessageManager $messageManager;

    /**
     * @param HttpRequest $request
     * @param Captcha $captchaModel
     * @param string[] $graphQlDefaultFormsRelation
     * @param ConfigProvider $configProvider
     */
    public function __construct(
        HttpRequest $request,
        Captcha $captchaModel,
        ConfigProvider $configProvider,
        array $graphQlDefaultFormsRelation,
        MessageManager $messageManager = null // TODO move to not optional
    ) {
        $this->request = $request;
        $this->captchaModel = $captchaModel;
        $this->configProvider = $configProvider;
        $this->graphQlDefaultFormsRelation = $graphQlDefaultFormsRelation;
        $this->messageManager = $messageManager ?? ObjectManager::getInstance()->get(MessageManager::class);
    }

    /**
     * @param Field $fieldInfo
     * @param ResolveInfo $resolveInfo
     * @return void
     * @throws GraphQlInputException
     */
    public function performValidation(Field $fieldInfo, ResolveInfo $resolveInfo): void
    {
        if (!$this->isMutation($resolveInfo)
            || !$this->shouldValidate($fieldInfo)
            || !$this->captchaModel->isNeedToShowCaptcha()
        ) {
            return;
        }

        $captchaToken = (string)$this->request->getHeader(self::CAPTCHA_HEADER_PARAM);
        $validation = $this->captchaModel->verify($captchaToken);

        if (!($validation['success'] ?? false)) {
            $this->messageManager->addErrorMessage(__($validation['error']));
            throw new GraphQlInputException(__($validation['error']));
        }
    }

    private function isMutation(ResolveInfo $resolveInfo): bool
    {
        return $resolveInfo->operation->operation === 'mutation';
    }

    private function shouldValidate(Field $fieldInfo): bool
    {
        return $this->isDefaultMutation($fieldInfo) || $this->isCustomMutation($fieldInfo);
    }

    private function isDefaultMutation(Field $fieldInfo): bool
    {
        $defaultActionName = $this->graphQlDefaultFormsRelation[$fieldInfo->getName()] ?? null;

        return $defaultActionName
            && in_array($defaultActionName, $this->configProvider->getEnabledDefaultForms(), true);
    }

    private function isCustomMutation(Field $fieldInfo): bool
    {
        return in_array($fieldInfo->getName(), $this->configProvider->getCustomUrls(), true);
    }
}
