<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Seo Rich Data Graphql by Amasty
 */

namespace Amasty\SeoRichDataGraphql\Model\Resolver;

use Amasty\Base\Model\Serializer;
use Amasty\SeoRichData\Helper\Config as ConfigHelper;
use Amasty\SeoRichData\Model\ConfigProvider;
use Amasty\SeoRichData\Model\JsonLd\ProductInfo;
use Magento\Catalog\Model\Product as ProductModel;
use Magento\Catalog\Model\ResourceModel\Product as ProductResource;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Framework\GraphQl\Query\Resolver\ContextInterface;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Store\Model\StoreManagerInterface;

class GetProductRichData implements ResolverInterface
{
    public const ATTRIBUTES = [
        'name',
        'price',
        'short_description',
        'description',
        'image'
    ];

    /**
     * @var ProductInfo
     */
    private $productInfo;

    /**
     * @var Serializer
     */
    private $serializer;

    /**
     * @var ConfigHelper
     */
    private $configHelper;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var ProductResource
     */
    private $productResource;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    public function __construct(
        ProductInfo $productInfo,
        Serializer $serializer,
        ConfigHelper $configHelper,
        StoreManagerInterface $storeManager,
        ProductResource $productResource,
        ConfigProvider $configProvider
    ) {
        $this->productInfo = $productInfo;
        $this->serializer = $serializer;
        $this->configHelper = $configHelper;
        $this->storeManager = $storeManager;
        $this->productResource = $productResource;
        $this->configProvider = $configProvider;
    }

    /**
     * @param Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return string
     * @throws GraphQlInputException
     */
    public function resolve(
        Field $field,
        $context,
        ResolveInfo $info,
        array $value = null,
        array $args = null
    ): string {
        $storeId = (int)$context->getExtensionAttributes()->getStore()->getId();
        if (!$this->configProvider->isEnabledForProduct($storeId)) {
            return '';
        }
        if (!isset($value['model'])) {
            throw new GraphQlInputException(__('"model" must be specified'));
        }
        $product = $value['model'];
        $this->storeManager->setCurrentStore($storeId);
        $product->addData($this->getAttributesData($product, $storeId));
        $productInfo = $this->productInfo->extract($product);

        return $this->serializer->serialize($productInfo);
    }

    private function getAttributesData(ProductModel $product, int $storeId): array
    {
        $attributesData = [];
        // It is needed to get attribute one by one.
        // There is a bug in getting attribute values if $attribute argument is an array.
        foreach ($this->getAttributes() as $attribute) {
            if (!$product->hasData($attribute)) {
                $attributesData[$attribute] = $this->productResource->getAttributeRawValue(
                    $product->getId(),
                    $attribute,
                    $storeId
                );
            }
        }

        return $attributesData;
    }

    private function getAttributes(): array
    {
        $attributes = self::ATTRIBUTES;
        if (null !== $this->configHelper->getBrandAttribute()) {
            $attributes[] = $this->configHelper->getBrandAttribute();
        }
        if (null !== $this->configHelper->getManufacturerAttribute()) {
            $attributes[] = $this->configHelper->getManufacturerAttribute();
        }

        return $attributes;
    }
}
