<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Seo Rich Data Graphql by Amasty
 */

namespace Amasty\SeoRichDataGraphql\Model\Resolver;

use Amasty\Base\Model\Serializer;
use Amasty\SeoRichData\Helper\Config as ConfigHelper;
use Magento\Catalog\Api\CategoryListInterface;
use Magento\Catalog\Api\Data\CategoryInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Query\Resolver\ContextInterface;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Store\Model\StoreManagerInterface;

class GetBreadcrumbs implements ResolverInterface
{
    /**
     * @var Serializer
     */
    private $serializer;

    /**
     * @var ConfigHelper
     */
    private $configHelper;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var CategoryListInterface
     */
    private $categoryRepository;

    /**
     * @var SearchCriteriaBuilder
     */
    private $searchCriteriaBuilder;

    public function __construct(
        Serializer $serializer,
        ConfigHelper $configHelper,
        StoreManagerInterface $storeManager,
        CategoryListInterface $categoryRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder
    ) {
        $this->serializer = $serializer;
        $this->configHelper = $configHelper;
        $this->storeManager = $storeManager;
        $this->categoryRepository = $categoryRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
    }

    /**
     * @param Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return string
     * @throws LocalizedException
     */
    public function resolve(
        Field $field,
        $context,
        ResolveInfo $info,
        array $value = null,
        array $args = null
    ): string {
        if (!$this->configHelper->isBreadcrumbsEnabled()) {
            return '';
        }
        if (!isset($value['path'])) {
            throw new LocalizedException(__('"path" should be specified'));
        }

        $breadcrumbs = '';
        $categoryPath = $value['path'];
        $pathCategoryIds = explode('/', $categoryPath);
        $parentCategoryIds = array_slice($pathCategoryIds, 2);
        if (count($parentCategoryIds)) {
            $categories = $this->getCategories($parentCategoryIds);
            $breadcrumbs = $this->serializer->serialize($this->getBreadcrumbs($categories));
        }

        return $breadcrumbs;
    }

    /**
     * @param array $parentCategoryIds
     *
     * @return CategoryInterface[]
     */
    private function getCategories(array $parentCategoryIds): array
    {
        $this->searchCriteriaBuilder
            ->addFilter('entity_id', $parentCategoryIds, 'in')
            ->addFilter(CategoryInterface::KEY_IS_ACTIVE, 1);
        $searchCriteria = $this->searchCriteriaBuilder->create();
        $searchResults = $this->categoryRepository->getList($searchCriteria);

        return $searchResults->getItems();
    }

    /**
     * @param CategoryInterface[] $categories
     *
     * @return array
     */
    private function getBreadcrumbs(array $categories): array
    {
        $data = [];
        // init Home breadcrumb.
        $items[] = [
            '@type' => 'ListItem',
            'position' => $position = 1,
            'item' => [
                '@id' => $this->storeManager->getStore()->getBaseUrl(),
                'name' => 'Home'
            ]
        ];
        // init categories breadcrumbs.
        foreach ($categories as $category) {
            $items[] = [
                '@type' => 'ListItem',
                'position' => ++$position,
                'item' => [
                    '@id' => $category->getUrl(),
                    'name' => $category->getName()
                ]
            ];
        }

        if (count($items) > 1) {
            if ($this->configHelper->sliceBreadcrumbs()) {
                $items = array_slice($items, -1, 1);
                if (isset($items[0])) {
                    $items[0]['position'] = 1;
                }
            }

            $data['breadcrumbs'] = [
                '@context' => 'https://schema.org',
                '@type' => 'BreadcrumbList',
                'itemListElement' => $items
            ];
        }

        return $data;
    }
}
