<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Gift Card Pro Hyva Compatibility
 */

declare(strict_types=1);

namespace Amasty\GiftCardHyvaCompatibility\ViewModel;

use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\View\Element\Block\ArgumentInterface;
use Amasty\GiftCard\Api\Data\GiftCardOptionInterface;
use Amasty\GiftCard\Api\Data\ImageInterface;
use Amasty\GiftCard\Model\Image\Repository;
use Amasty\GiftCard\Model\ConfigProvider;
use Magento\Framework\Registry;
use Amasty\GiftCard\Model\Image\ResourceModel\CollectionFactory;
use Amasty\GiftCard\Model\OptionSource\GiftCardOption;
use Amasty\GiftCard\Utils\FileUpload;
use Magento\Catalog\Model\Product;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Locale\ListsInterface;
use Magento\Framework\Pricing\PriceCurrencyInterface;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Framework\UrlInterface;
use Magento\Catalog\Model\ProductRepository;

class GiftCardViewModal implements ArgumentInterface
{
    /**
     * @var Product
     */
    protected $product = null;

    /**
     * @var Registry
     */
    private $coreRegistry = null;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var Repository
     */
    private $imageRepository;

    /**
     * @var GiftCardOption
     */
    private $giftCardOption;

    /**
     * @var ListsInterface
     */
    private $localeLists;

    /**
     * @var CollectionFactory
     */
    private $imageCollectionFactory;

    /**
     * @var FileUpload
     */
    private $fileUpload;

    /**
     * @var Json
     */
    private $jsonSerializer;

    /**
     * @var PriceCurrencyInterface
     */
    private $priceCurrency;

    /**
     * @var UrlInterface
     */
    private $urlBuilder;

    /**
     * @var ProductRepository
     */
    protected $productRepository;

    /**
     * @var array
     */
    private $amounts = [];

    /**
     * @param ConfigProvider $configProvider
     * @param Repository $imageRepository
     * @param GiftCardOption $giftCardOption
     * @param ListsInterface $localeLists
     * @param CollectionFactory $imageCollectionFactory
     * @param FileUpload $fileUpload
     * @param Json $jsonSerializer
     * @param PriceCurrencyInterface $priceCurrency
     * @param UrlInterface $urlBuilder
     * @param Registry $registry
     * @param ProductRepository $productRepository
     */
    public function __construct(
        ConfigProvider $configProvider,
        Repository $imageRepository,
        GiftCardOption $giftCardOption,
        ListsInterface $localeLists,
        CollectionFactory $imageCollectionFactory,
        FileUpload $fileUpload,
        Json $jsonSerializer,
        PriceCurrencyInterface $priceCurrency,
        UrlInterface $urlBuilder,
        Registry $registry,
        ProductRepository $productRepository
    ) {
        $this->configProvider = $configProvider;
        $this->imageRepository = $imageRepository;
        $this->giftCardOption = $giftCardOption;
        $this->localeLists = $localeLists;
        $this->imageCollectionFactory = $imageCollectionFactory;
        $this->fileUpload = $fileUpload;
        $this->jsonSerializer = $jsonSerializer;
        $this->priceCurrency = $priceCurrency;
        $this->coreRegistry = $registry;
        $this->urlBuilder = $urlBuilder;
        $this->productRepository = $productRepository;
    }

    /**
     * Get Product
     *
     * @param Int $productId
     * @return Product
     */
    public function getProduct($productId = null)
    {
        if ($productId) {
            return $this->productRepository->getById($productId);
        }

        if (!$this->product) {
            $this->product = $this->coreRegistry->registry('product');
        }

        return $this->product;
    }

    /**
     * Get Url
     *
     * @param Url $url
     * @return Url
     */
    public function getUrl($url): string
    {
        return $this->urlBuilder->getUrl($url);
    }

    /**
     * Get Images
     *
     * @return string
     */
    public function getImages(): string
    {
        $images = [];

        if ($productImagesId = $this->getProduct()->getAmGiftcardCodeImage()) {
            $productImagesId = explode(',', $productImagesId);
            $collection = $this->imageCollectionFactory->create()
                ->addFieldToFilter(ImageInterface::IMAGE_ID, ['in' => $productImagesId]);

            foreach ($collection->getItems() as $image) {
                try {
                    $images[] = [
                        'id' => $image->getImageId(),
                        'src' => $this->fileUpload->getImageUrl(
                            $image->getImagePath()
                        )
                    ];
                } catch (LocalizedException $e) {
                    continue;
                }
            }
        }

        return $this->jsonSerializer->serialize($images);
    }

    /**
     * Get Available Options
     *
     * @return string
     */
    public function getAvailableOptions(): string
    {
        $options = [];
        $allFields = $this->giftCardOption->toArray();
        $fields = $this->configProvider->getGiftCardFields();

        foreach ($allFields as $value => $label) {
            if (!in_array($value, $fields)) {
                continue;
            }

            $options[] = $value;
        }

        return $this->jsonSerializer->serialize($options);
    }

    /**
     * Get List Timezones
     *
     * @return string
     */
    public function getListTimezones(): string
    {
        $result = [];
        $allTimeZones = $this->localeLists->getOptionTimezones();
        $selectedTimeZones = $this->configProvider->getGiftCardTimezone();

        if (!$selectedTimeZones) {
            return $this->jsonSerializer->serialize($allTimeZones);
        }

        foreach ($allTimeZones as $timeZone) {
            if (in_array($timeZone['value'], $selectedTimeZones)) {
                $result[] = [
                    'value' => $timeZone['value'],
                    'label' => $timeZone['label']
                ];
            }
        }

        return $this->jsonSerializer->serialize($result);
    }

    /**
     * Get Allow Users Upload Images
     *
     * @return bool
     */
    public function getAllowUsersUploadImages(): bool
    {
        return $this->configProvider->isAllowUserImages();
    }

    /**
     * Get Tooltip Content
     *
     * @return string
     */
    public function getTooltipContent(): string
    {
        return $this->configProvider->getImageUploadTooltip();
    }

    /**
     * Get Default Values
     *
     * @return string
     */
    public function getDefaultValues(): string
    {
        return $this->jsonSerializer->serialize($this->getProduct()->getPreconfiguredValues()->getData());
    }

    /**
     * Get Custom ImageUrl
     *
     * @return string
     * @throws NoSuchEntityException
     */
    public function getCustomImageUrl(): string
    {
        if ($img = $this->getProduct()->getPreconfiguredValues()->getData(GiftCardOptionInterface::CUSTOM_IMAGE)) {
            return $this->fileUpload->getTempImgUrl($img);
        }

        return '';
    }

    /**
     * Check is Show Prices
     *
     * @return bool
     */
    public function isShowPrices(): bool
    {
        return $this->getProduct()->getAmAllowOpenAmount()
            || count((array)$this->getProduct()->getAmGiftcardPrices()) > 1;
    }

    /**
     * Check Single Price
     *
     * @return bool
     */
    public function isSinglePrice(): bool
    {
        $amountsCount = count((array)$this->getProduct()->getAmGiftcardPrices());
        $haveCustomAmount = $this->getProduct()->getAmAllowOpenAmount();

        return ($amountsCount === 1) && !$haveCustomAmount;
    }

    /**
     * Get Final Price
     *
     * @return string
     */
    public function getFinalPrice(): string
    {
        if ($this->isSinglePrice()) {
            $giftcardPrices = $this->getProduct()->getAmGiftcardPrices();

            return $giftcardPrices[0]['value'];
        }

        return '0';
    }

    /**
     * Get Predefined Amounts Json
     *
     * @param Int $productId
     * @return string
     */
    public function getPredefinedAmountsJson($productId = null): string
    {
        if (!empty($this->amounts)) {
            return $this->jsonSerializer->serialize($this->amounts);
        }

        foreach ((array)$this->getProduct($productId)->getAmGiftcardPrices() as $amount) {
            $this->amounts[] = [
                'value' => (float)$amount['value'],
                'convertValue' => $this->priceCurrency->convert((float)$amount['value']),
                'price' => $this->convertAndFormatCurrency((float)$amount['value'], false)
            ];
        }

        return $this->jsonSerializer->serialize($this->amounts);
    }

    /**
     * Get Currency Code
     *
     * @return string
     */
    public function getCurrencyCode(): string
    {
        $currency = $this->priceCurrency->getCurrency();

        return $currency->getCurrencyCode() ?: $currency->getCurrencySymbol();
    }

    /**
     * Get Fee Value Converted
     *
     * @return float
     */
    public function getFeeValueConverted(): float
    {
        return $this->priceCurrency->convert($this->getProduct()->getAmGiftcardFeeValue());
    }

    /**
     * Convert And Format Currency
     *
     * @param float $amount
     * @param bool $includeContainer
     * @return string
     */
    public function convertAndFormatCurrency(float $amount, bool $includeContainer = true): string
    {
        return $this->priceCurrency->convertAndFormat($amount, $includeContainer);
    }

    /**
     * Convert And Round Currency
     *
     * @param float $amount
     * @return float
     */
    public function convertAndRoundCurrency(float $amount): float
    {
        return $this->priceCurrency->convertAndRound($amount);
    }
}
