<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Gift Card Pro Hyva Compatibility
 */

declare(strict_types=1);

namespace Amasty\GiftCardHyvaCompatibility\ViewModel;

use Magento\Framework\View\Element\Block\ArgumentInterface;
use Magento\Framework\Serialize\SerializerInterface;
use Magento\Framework\Locale\Bundle\DataBundle;
use Magento\Framework\Locale\ResolverInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Stdlib\DateTime\TimezoneInterface;

/**
 * ViewModel to retrieve calendar config
 */
class Calendar implements ArgumentInterface
{
    /**
     * @var ResolverInterface
     */
    private ResolverInterface $localeResolver;

    /**
     * @var DataBundle
     */
    private DataBundle $dataBundle;

    /**
     * @var SerializerInterface
     */
    private SerializerInterface $serializer;

    /**
     * @var ScopeConfigInterface|MockObject
     */
    protected $scopeConfig;

    /**
     * @var TimezoneInterface
     */
    private TimezoneInterface $timezone;

    /**
     * @param SerializerInterface $serializer
     * @param DataBundle $dataBundle
     * @param ResolverInterface $localeResolver
     * @param ScopeConfigInterface $scopeConfig
     * @param TimezoneInterface $timezone
     */
    public function __construct(
        SerializerInterface $serializer,
        DataBundle $dataBundle,
        ResolverInterface $localeResolver,
        ScopeConfigInterface $scopeConfig,
        TimezoneInterface $timezone
    ) {
        $this->serializer = $serializer;
        $this->dataBundle = $dataBundle;
        $this->localeResolver = $localeResolver;
        $this->scopeConfig = $scopeConfig;
        $this->timezone = $timezone;
    }

    /**
     * Get translated calendar config json formatted
     *
     * @return string
     */
    public function getTranslatedCalendarConfigJson(): string
    {
        $localeData = (new DataBundle())->get($this->localeResolver->getLocale());
        $monthsData = $localeData['calendar']['gregorian']['monthNames'];
        $daysData = $localeData['calendar']['gregorian']['dayNames'];

        return $this->serializer->serialize(
            [
                'closeText' => __('Done'),
                'prevText' => __('Prev'),
                'nextText' => __('Next'),
                'currentText' => __('Today'),
                'monthNames' => array_values(iterator_to_array($monthsData['format']['wide'])),
                'monthNamesShort' => array_values(iterator_to_array($monthsData['format']['abbreviated'])),
                'dayNames' => array_values(iterator_to_array($daysData['format']['wide'])),
                'dayNamesShort' => array_values(iterator_to_array($daysData['format']['abbreviated'])),
                'dayNamesMin' => array_values(
                    iterator_to_array(($daysData['format']['short']) ?: $daysData['format']['abbreviated'])
                ),
            ]
        );
    }

    /**
     * Return Date Format
     *
     * @return string
     */
    public function getDateFormat(): string
    {
        return $this->timezone->getDateFormat();
    }

    /**
     * Return first day of the week
     *
     * @return int
     */
    public function getFirstDay(): int
    {
        return (int)$this->scopeConfig->getValue(
            'general/locale/firstday',
            \Magento\Store\Model\ScopeInterface::SCOPE_STORE
        );
    }
}
