<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Gift Card Premium Functionality for Magento 2 (System)
 */

namespace Amasty\GiftCardPremiumFunctionality\Controller\Account;

use Amasty\GiftCardAccount\Api\GiftCardAccountCommandManagementInterface;
use Amasty\GiftCardAccount\Controller\Account\AddCard;
use Amasty\GiftCardAccount\Model\GiftCardAccountFormatter;
use Amasty\GiftCardPremiumFunctionality\Model\ConfigProvider;
use Magento\Customer\Model\Session as CustomerSession;
use Magento\Framework\App\Action\Action;
use Magento\Framework\App\Action\Context;
use Magento\Framework\App\Action\HttpPostActionInterface;
use Magento\Framework\Controller\ResultFactory;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Exception\NotFoundException;

class Redeem extends Action implements HttpPostActionInterface
{
    /**
     * @var GiftCardAccountCommandManagementInterface
     */
    private $giftCardAccountCommandManagement;

    /**
     * @var CustomerSession
     */
    private $session;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var GiftCardAccountFormatter
     */
    private $accountFormatter;

    public function __construct(
        Context $context,
        CustomerSession $session,
        GiftCardAccountCommandManagementInterface $giftCardAccountCommandManagement,
        ConfigProvider $configProvider,
        GiftCardAccountFormatter $accountFormatter
    ) {
        parent::__construct($context);
        $this->giftCardAccountCommandManagement = $giftCardAccountCommandManagement;
        $this->session = $session;
        $this->configProvider = $configProvider;
        $this->accountFormatter = $accountFormatter;
    }

    public function execute()
    {
        $resultJson = $this->resultFactory->create(ResultFactory::TYPE_JSON);
        try {
            if (!$this->configProvider->isEnabled()) {
                throw new NotFoundException(__('Invalid Request'));
            }
            if (!$this->session->isLoggedIn()) {
                throw new LocalizedException(__('The session has expired. Please refresh the page.'));
            }
            $customerId = (int)$this->session->getCustomerId();
            $code = trim($this->getRequest()->getParam(AddCard::AM_GIFTCARD_CODE_KEY, ''));
            $account = $this->giftCardAccountCommandManagement->redeemToAmStoreCredit($code, $customerId);
            $response = [
                'message' => __(
                    'Gift Card "%1" was successfully redeemed for store credit.',
                    $account->getCodeModel()->getCode()
                ),
                'data' => $this->accountFormatter->getFormattedData($account)
            ];
        } catch (NoSuchEntityException $e) {
            $response = [
                'message' => __('Wrong Gift Card code.'),
                'error'   => true
            ];
        } catch (NotFoundException | LocalizedException $e) {
            $response = [
                'message' => $e->getMessage(),
                'error'   => true
            ];
        } catch (\Exception $e) {
            $response = [
                'message' => __('Something went wrong.'),
                'error'   => true
            ];
        }

        return $resultJson->setData($response);
    }
}
