<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Gift Card GraphQL for Magento 2 (System)
 */

namespace Amasty\GiftCardGraphQl\Test\GraphQl;

use Amasty\GiftCardAccount\Api\GiftCardAccountManagementInterface;
use Magento\GraphQl\Quote\GetMaskedQuoteIdByReservedOrderId;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\Quote\Model\GetQuoteByReservedOrderId;
use Magento\TestFramework\TestCase\GraphQlAbstract;

class RemoveAmGiftCardFromCartTest extends GraphQlAbstract
{
    private const MAIN_RESPONSE_KEY = 'removeAmGiftCardFromCart';

    /**
     * @var GetMaskedQuoteIdByReservedOrderId
     */
    private $getMaskedQuoteIdByReservedOrderId;

    /**
     * @var GiftCardAccountManagementInterface
     */
    private $giftCardAccountManagement;

    /**
     * @var GetQuoteByReservedOrderId
     */
    private $getQuoteByReservedOrderId;

    protected function setUp(): void
    {
        parent::setUp();

        $objectManager = Bootstrap::getObjectManager();
        $this->getMaskedQuoteIdByReservedOrderId = $objectManager->get(GetMaskedQuoteIdByReservedOrderId::class);
        $this->getQuoteByReservedOrderId = $objectManager->get(GetQuoteByReservedOrderId::class);
        $this->giftCardAccountManagement = $objectManager->get(GiftCardAccountManagementInterface::class);
    }

    /**
     * @group amasty_gift_card
     *
     * @magentoConfigFixture base_website amgiftcard/general/active 1
     *
     * @magentoApiDataFixture Amasty_GiftCardGraphQl::Test/GraphQl/_files/am_customer_cart_with_product.php
     */
    public function testRemoveAmGiftCardFromCart(): void
    {
        $cardAccountCode = 'AM_TEST_7';
        $reservedOrderId = 'am_gift_customer_quote';

        $quoteId = $this->getQuoteByReservedOrderId->execute($reservedOrderId)->getId();
        $maskedQuote = $this->getMaskedQuoteIdByReservedOrderId->execute($reservedOrderId);

        $giftCardAccountResponse = $this->giftCardAccountManagement->applyGiftCardAccountToCart(
            $quoteId,
            $cardAccountCode
        );
        $resultApplied = $giftCardAccountResponse->getMessages();
        $this->assertEquals($resultApplied[0]->getType(), 'success');

        $variables = [
            'amGiftCardCode' => $cardAccountCode,
            'cartId' => $maskedQuote
        ];

        $assertArray = [
            'cart' => [
                'applied_am_gift_cards' => [],
                'prices' => [
                    'grand_total' => [
                        'value' => 40
                    ]
                ]
            ]
        ];

        $response = $this->graphQlMutation($this->getQuery(), $variables);

        $this->assertArrayHasKey(self::MAIN_RESPONSE_KEY, $response);
        $this->assertResponseFields($response[self::MAIN_RESPONSE_KEY], $assertArray);
    }

    private function getQuery(): string
    {
        return <<<'QUERY'
mutation RemoveAmGiftCardFromCart (
    $amGiftCardCode: String!,
    $cartId: String!
    ){
    removeAmGiftCardFromCart (
      input: {
        am_gift_card_code:$amGiftCardCode,
        cart_id:$cartId
      }
    ) {
        cart {
          applied_am_gift_cards {
            code
          }
          prices {
            grand_total {
              value
            }
          }
        }
    }
}
QUERY;
    }
}
