<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Gift Card GraphQL for Magento 2 (System)
 */

namespace Amasty\GiftCardGraphQl\Test\GraphQl;

use Magento\Integration\Api\CustomerTokenServiceInterface;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\GraphQlAbstract;

class GetAmUserGiftCardAccountTest extends GraphQlAbstract
{
    private const MAIN_RESPONSE_KEY = 'amUserGiftCardAccount';
    public const CUSTOMER = 'customer_with_addresses@test.com';
    public const CUSTOMER_PASS = 'password';

    /**
     * @var CustomerTokenServiceInterface
     */
    private $customerTokenService;

    protected function setUp(): void
    {
        parent::setUp();

        $this->customerTokenService = Bootstrap::getObjectManager()->get(CustomerTokenServiceInterface::class);
    }

    /**
     * @group amasty_gift_card
     *
     * @magentoConfigFixture base_website amgiftcard/general/active 1
     *
     * @magentoApiDataFixture Amasty_GiftCardGraphQl::Test/GraphQl/_files/am_gift_card_code_accounts.php
     */
    public function testGetAmUserGiftCardAccount(): void
    {
        $assertArrayItemOne = [
            'code' => 'AM_TEST_4',
            'current_balance' => [
                'currency' => 'USD',
                'value' => 900
            ],
            'expiration_date' => 'Unlimited',
            'status' => 'Active'
        ];
        $assertArrayItemTwo = [
            'code' => 'AM_TEST_8',
            'current_balance' => [
                'currency' => 'USD',
                'value' => 1200
            ],
            'expiration_date' => 'Unlimited',
            'status' => 'Inactive'
        ];

        $response = $this->graphQlQuery($this->getQuery(), [], '', $this->getHeader());

        $this->assertArrayHasKey(self::MAIN_RESPONSE_KEY, $response);
        $this->assertCount(2, $response[self::MAIN_RESPONSE_KEY]);

        foreach ($response[self::MAIN_RESPONSE_KEY] as $item) {
            if ($item['code'] == 'AM_TEST_4') {
                $this->assertResponseFields($item, $assertArrayItemOne);
                continue;
            }
            $this->assertResponseFields($item, $assertArrayItemTwo);
        }
    }

    private function getQuery(): string
    {
        return <<<QUERY
query {
    amUserGiftCardAccount {
        code
        current_balance {
          currency
          value
        }
        expiration_date
        status
    }
}
QUERY;
    }

    /**
     * @param string $userName
     * @param string $password
     *
     * @return string[]
     */
    private function getHeader(
        string $userName = self::CUSTOMER,
        string $password = self::CUSTOMER_PASS
    ): array {
        $customerToken = $this->customerTokenService->createCustomerAccessToken($userName, $password);
        return ['Authorization' => 'Bearer ' . $customerToken];
    }
}
