<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Gift Card GraphQL for Magento 2 (System)
 */

namespace Amasty\GiftCardGraphQl\Test\GraphQl;

use Amasty\GiftCard\Api\ImageRepositoryInterface;
use Magento\GraphQl\Quote\GetMaskedQuoteIdByReservedOrderId;
use Magento\Integration\Api\CustomerTokenServiceInterface;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\GraphQlAbstract;

class AddAmGiftCardProductsToCartTest extends GraphQlAbstract
{
    private const MAIN_RESPONSE_KEY = 'addAmGiftCardProductsToCart';

    /**
     * @var GetMaskedQuoteIdByReservedOrderId
     */
    private $getMaskedQuoteIdByReservedOrderId;

    /**
     * @var ImageRepositoryInterface
     */
    private $imageRepository;

    protected function setUp(): void
    {
        parent::setUp();

        $objectManager = Bootstrap::getObjectManager();
        $this->getMaskedQuoteIdByReservedOrderId = $objectManager->get(GetMaskedQuoteIdByReservedOrderId::class);
        $this->imageRepository = $objectManager->get(ImageRepositoryInterface::class);
    }

    /**
     * @group amasty_gift_card
     *
     * @magentoConfigFixture base_website amgiftcard/general/active 1
     *
     * @magentoApiDataFixture Amasty_GiftCardGraphQl::Test/GraphQl/_files/am_customer_cart_with_product.php
     * @magentoApiDataFixture Amasty_GiftCard::Test/Integration/_files/giftcard_product_virtual_with_amounts.php
     */
    public function testAddAmGiftCardProductsToCart(): void
    {
        $maskedQuote = $this->getMaskedQuoteIdByReservedOrderId->execute('am_gift_customer_quote');

        $imageId = 0;
        $images = $this->imageRepository->getList();
        foreach ($images as $image) {
            if ($image->getTitle() == 'Test Image') {
                $imageId = $image->getImageId();
            }
        }

        $variables = [
            'cartId' => $maskedQuote,
            'sku' => 'am_giftcard_with_amounts',
            'qty' => 2,
            'amGiftcardAmount' => 10,
            'amGiftcardImage' => $imageId,
            'amGiftcardRecipientEmail' => 'test_recipient@amasty.com',
            'amGiftcardRecipientName' => 'Recipient Test Amasty',
            'amGiftcardSenderName' => 'Sender Test Amasty'
        ];

        $assertArray = [
            'cart' => [
                'items' => [
                    [
                        'product' => [
                            'sku' => 'simple_20'
                        ],
                        'prices' => [
                            'price' => [
                                'value' => 20
                            ]
                        ],
                        'quantity' => 2
                    ],
                    [
                        'product' => [
                            'sku' => 'am_giftcard_with_amounts'
                        ],
                        'prices' => [
                            'price' => [
                                'value' => 10
                            ]
                        ],
                        'quantity' => 2
                    ]
                ],
                'prices' => [
                    'grand_total' => [
                        'value' => 60
                    ]
                ]
            ]
        ];

        $response = $this->graphQlMutation($this->getQuery(), $variables);

        $this->assertArrayHasKey(self::MAIN_RESPONSE_KEY, $response);
        $this->assertResponseFields($response[self::MAIN_RESPONSE_KEY], $assertArray);
    }

    private function getQuery(): string
    {
        return <<<'QUERY'
mutation AddAmGiftCardProductsToCart (
    $cartId: String!,
    $sku: String!,
    $qty: Float!,
    $amGiftcardAmount: Float,
    $amGiftcardRecipientEmail: String,
    $amGiftcardImage: Int,
    $amGiftcardRecipientName: String,
    $amGiftcardSenderName: String
) {
    addAmGiftCardProductsToCart (
        input: {
            cart_id:$cartId,
            cart_items: {
                data: {
                    sku: $sku
                    quantity: $qty
                }
                gift_card_options: {
                    am_giftcard_amount: $amGiftcardAmount,
                    am_giftcard_recipient_email: $amGiftcardRecipientEmail,
                    am_giftcard_image: $amGiftcardImage,
                    am_giftcard_recipient_name: $amGiftcardRecipientName,
                    am_giftcard_sender_name: $amGiftcardSenderName
                }
            }
        }
    ) {
        cart {
            items {
                product {
                    sku
                }
                prices {
                    price {
                        value
                    }
                }
                quantity
            }
            prices {
                grand_total {
                    value
                }
            }
        }
    }
}
QUERY;
    }
}
