<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Gift Card Codes Auto-Generation
 */

namespace Amasty\GiftCardCodesAutoGeneration\Model\Queue;

use Amasty\Base\Model\Serializer;
use Amasty\GiftCard\Api\Data\CodeInterface;
use Amasty\GiftCard\Api\Data\CodePoolInterface;
use Amasty\GiftCard\Model\Code\ResourceModel\Code;
use Amasty\GiftCard\Model\OptionSource\Status;
use Amasty\GiftCardCodesAutoGeneration\Model\ResourceModel\Queue\GiftCardMessagesQueue;
use Magento\Framework\MessageQueue\PublisherInterface;
use Magento\MysqlMq\Model\QueueManagement;

class Publisher
{
    public const TOPIC_NAME = 'amgcard.codepool.autogeneration';

    /**
     * @var PublisherInterface
     */
    private $publisher;

    /**
     * @var Serializer
     */
    private $serializer;

    /**
     * @var GiftCardMessagesQueue
     */
    private $queueResource;

    /**
     * @var Code
     */
    private $codesResource;

    public function __construct(
        PublisherInterface $publisher,
        Serializer $serializer,
        GiftCardMessagesQueue $queueResource,
        Code $codesResource
    ) {
        $this->publisher = $publisher;
        $this->serializer = $serializer;
        $this->queueResource = $queueResource;
        $this->codesResource = $codesResource;
    }

    public function publish(CodePoolInterface $codePool): void
    {
        if (!$this->canPublish($codePool)) {
            return;
        }

        $this->publisher->publish(
            self::TOPIC_NAME,
            (string)$codePool->getCodePoolId()
        );
    }

    private function canPublish(CodePoolInterface $codePool): bool
    {
        $autoGenerationData = $codePool->getExtensionAttributes()->getAutoGenerationData();
        if (!$autoGenerationData || !$autoGenerationData->isEnabled()) {
            return false;
        }

        if ($this->checkAwaitingMessages($codePool) || $this->checkThresholdReached($codePool)) {
            return false;
        }

        return true;
    }

    private function checkAwaitingMessages(CodePoolInterface $codePool): bool
    {
        $codePoolId = $codePool->getCodePoolId();
        $awaitingMessages = $this->queueResource->getGiftCardMessages();
        foreach ($awaitingMessages as $message) {
            $body = (int)$this->serializer->unserialize($message[QueueManagement::MESSAGE_BODY]);
            if ($body === $codePoolId) {
                return true;
            }
        }

        return false;
    }

    private function checkThresholdReached(CodePoolInterface $codePool): bool
    {
        $autoGenerationData = $codePool->getExtensionAttributes()->getAutoGenerationData();
        $codesCount = $this->codesResource->retrieveFreeCodesCount($codePool->getCodePoolId());

        return $codesCount > $autoGenerationData->getThresholdQty();
    }
}
