<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Gift Card Codes Auto-Generation
 */

namespace Amasty\GiftCardCodesAutoGeneration\Model\Queue;

use Amasty\GiftCard\Api\Data\CodePoolInterface;
use Amasty\GiftCard\Model\Code\CodeGeneratorManagement;
use Amasty\GiftCard\Model\Code\CodeTemplateValidator;
use Amasty\GiftCard\Model\CodePool\ResourceModel\CodePool;
use Amasty\GiftCardCodesAutoGeneration\Model\ConfigProvider;
use Amasty\GiftCardCodesAutoGeneration\Model\Email\SendAutoGenerationAlert;
use Magento\Framework\Exception\LocalizedException;
use Psr\Log\LoggerInterface;

class Consumer
{
    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var SendAutoGenerationAlert
     */
    private $sendAutoGenerationAlert;

    /**
     * @var CodeGeneratorManagement
     */
    private $codeGeneratorManagement;

    /**
     * @var CodeTemplateValidator
     */
    private $codeTemplateValidator;

    /**
     * @var CodePool
     */
    private $codePoolResource;

    public function __construct(
        LoggerInterface $logger,
        ConfigProvider $configProvider,
        SendAutoGenerationAlert $sendAutoGenerationAlert,
        CodeGeneratorManagement $codeGeneratorManagement,
        CodeTemplateValidator $codeTemplateValidator,
        CodePool $codePoolResource
    ) {
        $this->logger = $logger;
        $this->configProvider = $configProvider;
        $this->sendAutoGenerationAlert = $sendAutoGenerationAlert;
        $this->codeGeneratorManagement = $codeGeneratorManagement;
        $this->codeTemplateValidator = $codeTemplateValidator;
        $this->codePoolResource = $codePoolResource;
    }

    public function process(string $codePoolId): void
    {
        try {
            $this->execute((int)$codePoolId);
        } catch (\Exception $e) {
            $this->logger->critical($e->getMessage());
            if ($this->configProvider->isNotifyAutoGeneration()) {
                $this->sendAutoGenerationAlert->execute((int)$codePoolId, $e->getMessage());
            }
        }
    }

    /**
     * @throws LocalizedException
     */
    private function execute(int $codePoolId): void
    {
        $codePoolData = $this->getCodePoolData($codePoolId);

        $qty = (float)($codePoolData[CodePoolInterface::QTY] ?? .0);
        $template = $codePoolData[CodePoolInterface::TEMPLATE] ?? '';
        $warningMessage = false;

        $availableQty = $this->codeTemplateValidator->getTemplateAvailableCodesQty($template);
        if ($qty > $availableQty) {
            $qty = $availableQty;
            $warningMessage = true;
        }

        $this->codeGeneratorManagement->generateCodesByTemplate(
            $codePoolId,
            $template,
            (int)$qty
        );

        if ($warningMessage) {
            throw new LocalizedException(__(
                'The last %1 codes were generated. The threshold of unique code values has been reached,'
                . ' please change codes template.',
                $qty
            ));
        }
    }

    /**
     * @param int $codePoolId
     *
     * @return array [qty, template]
     */
    private function getCodePoolData(int $codePoolId): array
    {
        $connection = $this->codePoolResource->getConnection();
        $select = $connection->select()->from(
            $this->codePoolResource->getMainTable(),
            [CodePoolInterface::QTY, CodePoolInterface::TEMPLATE]
        )->where(
            CodePoolInterface::CODE_POOL_ID . ' = ?',
            $codePoolId
        );
        $result = $connection->fetchAssoc($select);

        return array_pop($result);
    }
}
