<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Gift Card Codes Auto-Generation
 */

namespace Amasty\GiftCardCodesAutoGeneration\Model\Email;

use Amasty\GiftCard\Model\CodePool\Repository;
use Amasty\GiftCardCodesAutoGeneration\Model\ConfigProvider;
use Magento\Framework\App\Area;
use Magento\Framework\App\State as AppState;
use Magento\Framework\Mail\Template\TransportBuilder;
use Magento\Store\Model\StoreManagerInterface;
use Psr\Log\LoggerInterface;

class SendAutoGenerationAlert
{
    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var TransportBuilder
     */
    private $transportBuilder;

    /**
     * @var Repository
     */
    private $codePoolRepository;

    /**
     * @var AppState
     */
    private $appState;

    public function __construct(
        ConfigProvider $configProvider,
        LoggerInterface $logger,
        StoreManagerInterface $storeManager,
        TransportBuilder $transportBuilder,
        Repository $codePoolRepository,
        AppState $appState
    ) {
        $this->configProvider = $configProvider;
        $this->logger = $logger;
        $this->storeManager = $storeManager;
        $this->transportBuilder = $transportBuilder;
        $this->codePoolRepository = $codePoolRepository;
        $this->appState = $appState;
    }

    public function execute(int $codePoolId, string $message): void
    {
        try {
            $storeId = (int)$this->storeManager->getStore()->getId();
            $sendTo = $this->configProvider->getAutoGenerationEmailRecipients($storeId);
            $templateIdentifier = $this->configProvider->getAutoGenerationEmailTemplate();
            $sender = $this->configProvider->getAutoGenerationEmailSender();

            $codePool = $this->codePoolRepository->getById($codePoolId);

            foreach ($sendTo as $receiver) {
                $this->transportBuilder->setTemplateIdentifier($templateIdentifier)
                    ->setTemplateOptions(['area' => Area::AREA_FRONTEND, 'store' => $storeId])
                    ->setTemplateVars(['code_pool_name' => $codePool->getTitle(), 'error' => $message])
                    ->setFromByScope($sender, $storeId);

                $this->transportBuilder->addTo($receiver);
                $transport = $this->appState->emulateAreaCode(
                    Area::AREA_FRONTEND,
                    [$this->transportBuilder, 'getTransport']
                );
                $transport->sendMessage();
            }
        } catch (\Exception $e) {
            $this->logger->critical($e);
        }
    }
}
