<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package GDPR GraphQl for Magento 2 (System)
 */

namespace Amasty\GdprGraphQl\Test\GraphQl;

use Amasty\Gdpr\Api\DeleteRequestRepositoryInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Framework\Exception\AuthenticationException;
use Magento\Integration\Api\CustomerTokenServiceInterface;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\GraphQlAbstract;

class RequestAmGdprAccountDeletionTest extends GraphQlAbstract
{
    public const MAIN_QUERY_KEY = 'requestAmGdprAccountDeletion';
    public const CUSTOMER = 'new_customer@example.com';
    public const CUSTOMER_PASS = 'Qwert12345';

    /**
     * @var DeleteRequestRepositoryInterface
     */
    private $deleteRequestRepository;

    /**
     * @var SearchCriteriaBuilder
     */
    private $searchCriteriaBuilder;

    /**
     * @var CustomerRepositoryInterface
     */
    private $customerRepository;

    /**
     * @var CustomerTokenServiceInterface
     */
    private $customerTokenService;

    protected function setUp(): void
    {
        $objectManager = Bootstrap::getObjectManager();
        $this->deleteRequestRepository = $objectManager->create(DeleteRequestRepositoryInterface::class);
        $this->searchCriteriaBuilder = $objectManager->create(SearchCriteriaBuilder::class);
        $this->customerRepository = $objectManager->create(CustomerRepositoryInterface::class);
        $this->customerTokenService = $objectManager->get(CustomerTokenServiceInterface::class);
    }

    /**
     * @group amasty_gdpr
     *
     * @magentoConfigFixture base_website amasty_gdpr/customer_access_control/delete 1
     *
     * @magentoDataFixture Magento/Customer/_files/new_customer.php
     */
    public function testRequestAmGdprAccountDeletion(): void
    {
        $fields = [
            'error' => false,
            'message' => "Thank you, your account delete request was recorded."
        ];

        $query = $this->getQuery(self::CUSTOMER_PASS);
        $response = $this->graphQlMutation($query, [], '', $this->getHeader());

        $this->assertArrayHasKey(self::MAIN_QUERY_KEY, $response);
        $this->assertResponseFields($response[self::MAIN_QUERY_KEY], $fields);

        $customerId = $this->customerRepository->get(self::CUSTOMER)->getId();
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter('customer_id', (int)$customerId)
            ->addFilter('got_from', 'customer_request')
            ->create();
        $deleteLogs = $this->deleteRequestRepository->getList($searchCriteria)->getItems();

        $this->assertArrayHasKey(
            0,
            $deleteLogs,
            'Delete request did not added to GDPR delete request table.'
        );

        foreach ($deleteLogs as $log) {
            $this->assertIsObject($log);
        }
    }

    /**
     * Returns GraphQl query string
     */
    private function getQuery(string $password): string
    {
        return <<<MUTATION
mutation {
    requestAmGdprAccountDeletion(
        input: {
            password: "$password"
    }) {
        error
        message
    }
}
MUTATION;
    }

    /**
     * @param string $userName
     * @param string $password
     *
     * @return string[]
     * @throws AuthenticationException
     */
    private function getHeader(
        string $userName = self::CUSTOMER,
        string $password = self::CUSTOMER_PASS
    ): array {
        $customerToken = $this->customerTokenService->createCustomerAccessToken($userName, $password);
        return ['Authorization' => 'Bearer ' . $customerToken];
    }
}
