<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package GDPR GraphQl for Magento 2 (System)
 */

namespace Amasty\GdprGraphQl\Test\GraphQl;

use Amasty\Gdpr\Api\WithConsentRepositoryInterface;
use Amasty\Gdpr\Model\Repository\WithConsentRepository;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\GraphQlAbstract;

class GiveAmGdprConsentTest extends GraphQlAbstract
{
    public const MAIN_QUERY_KEY = 'giveAmGdprConsent';

    /**
     * @var WithConsentRepositoryInterface
     */
    private $withConsentRepository;

    /**
     * @var SearchCriteriaBuilder
     */
    private $searchCriteriaBuilder;

    protected function setUp(): void
    {
        $objectManager = Bootstrap::getObjectManager();
        $this->withConsentRepository = $objectManager->create(WithConsentRepository::class);
        $this->searchCriteriaBuilder = $objectManager->create(SearchCriteriaBuilder::class);
    }

    /**
     * @group amasty_gdpr
     *
     * @magentoApiDataFixture Amasty_GdprGraphQl::Test/GraphQl/_files/create_privacy_policy.php
     * @magentoApiDataFixture Amasty_GdprGraphQl::Test/GraphQl/_files/create_consent.php
     */
    public function testRequestAmGdprAccountDeletion(): void
    {
        $fields = [
            'error' => false,
            'message' => "Changes has been saved"
        ];

        $variables = [
            'consentCode' => 'privacy_test_graphql_checkbox',
            'consentFrom' => 'registration',
            'hasAgreement' => true
        ];

        $query = $this->getQuery();
        $response = $this->graphQlMutation($query, $variables);

        $this->assertArrayHasKey(self::MAIN_QUERY_KEY, $response);
        $this->assertResponseFields($response[self::MAIN_QUERY_KEY], $fields);

        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter('customer_id', 0)
            ->addFilter('policy_version', 'v2.0')
            ->addFilter('got_from', 'registration')
            ->addFilter('consent_code', 'privacy_test_graphql_checkbox')
            ->addFilter('action', 1)
            ->create();
        $consentLogs = $this->withConsentRepository->getList($searchCriteria)->getItems();

        $this->assertArrayHasKey(
            0,
            $consentLogs,
            'Consent give did not added to GDPR consent log.'
        );

        foreach ($consentLogs as $log) {
            $this->assertIsObject($log);
        }
    }

    /**
     * Returns GraphQl query string
     */
    private function getQuery(): string
    {
        return <<<'MUTATION'
mutation GiveAmGdprConsent(
    $consentCode: String!,
    $consentFrom: String!,
    $hasAgreement: Boolean!
){
    giveAmGdprConsent(
        input: {
            consents: {
                consent_code: $consentCode,
                consent_from: $consentFrom,
                has_agreement: $hasAgreement
            }
    }) {
        error
        message
    }
}
MUTATION;
    }
}
