<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package GDPR GraphQl for Magento 2 (System)
 */

namespace Amasty\GdprGraphQl\Test\GraphQl;

use Amasty\Gdpr\Api\ActionLogRepositoryInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Framework\Exception\AuthenticationException;
use Magento\Integration\Api\CustomerTokenServiceInterface;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\GraphQlAbstract;

class AnonymiseAmGdprCustomerTest extends GraphQlAbstract
{
    public const MAIN_QUERY_KEY = 'anonymiseAmGdprCustomer';
    public const CUSTOMER = 'new_customer@example.com';
    public const CUSTOMER_PASS = 'Qwert12345';

    /**
     * @var ActionLogRepositoryInterface
     */
    private $actionLogRepository;

    /**
     * @var SearchCriteriaBuilder
     */
    private $searchCriteriaBuilder;

    /**
     * @var CustomerRepositoryInterface
     */
    private $customerRepository;

    /**
     * @var CustomerTokenServiceInterface
     */
    private $customerTokenService;

    protected function setUp(): void
    {
        $objectManager = Bootstrap::getObjectManager();
        $this->actionLogRepository = $objectManager->create(ActionLogRepositoryInterface::class);
        $this->searchCriteriaBuilder = $objectManager->create(SearchCriteriaBuilder::class);
        $this->customerRepository = $objectManager->create(CustomerRepositoryInterface::class);
        $this->customerTokenService = $objectManager->get(CustomerTokenServiceInterface::class);
    }

    /**
     * @group amasty_gdpr
     *
     * @magentoConfigFixture base_website amasty_gdpr/customer_access_control/anonymize 1
     *
     * @magentoDataFixture Magento/Customer/_files/new_customer.php
     */
    public function testAnonymiseAmGdprCustomer(): void
    {
        $fields = [
            'error' => false,
            'message' => "Anonymisation was successful."
        ];

        $query = $this->getQuery(self::CUSTOMER_PASS);
        $response = $this->graphQlMutation($query, [], '', $this->getHeader());

        $this->assertArrayHasKey(self::MAIN_QUERY_KEY, $response);
        $this->assertResponseFields($response[self::MAIN_QUERY_KEY], $fields);

        $customerId = $this->customerRepository->get(self::CUSTOMER)->getId();
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter('customer_id', (int)$customerId)
            ->addFilter('action', 'data_anonymised_by_customer')
            ->create();
        $anonymiseLogs = $this->actionLogRepository->getList($searchCriteria)->getItems();

        $this->assertArrayHasKey(
            0,
            $anonymiseLogs,
            'Anonymise action did not added to GDPR actions log.'
        );

        foreach ($anonymiseLogs as $log) {
            $this->assertIsObject($log);
        }
    }

    /**
     * Returns GraphQl query string
     */
    private function getQuery(string $password): string
    {
        return <<<MUTATION
mutation {
    anonymiseAmGdprCustomer(
        input: {
            password: "$password"
    }) {
        error
        message
    }
}
MUTATION;
    }

    /**
     * @param string $userName
     * @param string $password
     *
     * @return string[]
     * @throws AuthenticationException
     */
    private function getHeader(
        string $userName = self::CUSTOMER,
        string $password = self::CUSTOMER_PASS
    ): array {
        $customerToken = $this->customerTokenService->createCustomerAccessToken($userName, $password);
        return ['Authorization' => 'Bearer ' . $customerToken];
    }
}
