<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package GDPR GraphQl for Magento 2 (System)
 */

namespace Amasty\GdprGraphQl\Test\GraphQl;

use Amasty\Gdpr\Api\ConsentRepositoryInterface;
use Magento\Framework\Exception\AuthenticationException;
use Magento\Integration\Api\CustomerTokenServiceInterface;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\GraphQlAbstract;

class AmGetGdprPolicyTextTest extends GraphQlAbstract
{
    public const MAIN_QUERY_KEY = 'amGetGdprPolicyText';
    public const CUSTOMER = 'new_customer@example.com';
    public const CUSTOMER_PASS = 'Qwert12345';

    /**
     * @var ConsentRepositoryInterface
     */
    private $consentRepository;

    /**
     * @var CustomerTokenServiceInterface
     */
    private $customerTokenService;

    protected function setUp(): void
    {
        $objectManager = Bootstrap::getObjectManager();
        $this->consentRepository = $objectManager->get(ConsentRepositoryInterface::class);
        $this->customerTokenService = $objectManager->get(CustomerTokenServiceInterface::class);
    }

    /**
     * @group amasty_gdpr
     *
     * @magentoConfigFixture base_website amasty_gdpr/general/display_pp_popup 1
     *
     * @magentoDataFixture Magento/Customer/_files/new_customer.php
     *
     * @magentoApiDataFixture Amasty_GdprGraphQl::Test/GraphQl/_files/create_privacy_policy.php
     * @magentoApiDataFixture Amasty_GdprGraphQl::Test/GraphQl/_files/create_consent.php
     */
    public function testAmGetGdprPolicyText(): void
    {
        $consentId = $this->consentRepository->getByConsentCode('privacy_test_graphql_checkbox')->getConsentId();

        $fields = [
            'content' => 'Specific Test GraphQl Policy Content <h4><strong>Contacting Us</strong></h4>',
            'need_show' => true,
            'title' => 'Privacy Policy',
            'version' => 'v2.0',
            'version_changed' => false
        ];

        $query = $this->getQuery((int)$consentId);
        $response = $this->graphQlQuery($query, [], '', $this->getHeader());

        $this->assertArrayHasKey(self::MAIN_QUERY_KEY, $response);
        $this->assertResponseFields($response[self::MAIN_QUERY_KEY], $fields);
    }

    /**
     * Returns GraphQl query string
     */
    private function getQuery(int $consentId): string
    {
        return <<<QUERY
query {
    amGetGdprPolicyText(id: $consentId) {
        content
        need_show
        title
        version
        version_changed
    }
}
QUERY;
    }

    /**
     * @param string $userName
     * @param string $password
     *
     * @return string[]
     * @throws AuthenticationException
     */
    private function getHeader(
        string $userName = self::CUSTOMER,
        string $password = self::CUSTOMER_PASS
    ): array {
        $customerToken = $this->customerTokenService->createCustomerAccessToken($userName, $password);
        return ['Authorization' => 'Bearer ' . $customerToken];
    }
}
