<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package GDPR GraphQl for Magento 2 (System)
 */

namespace Amasty\GdprGraphQl\Test\GraphQl;

use Magento\Framework\Exception\AuthenticationException;
use Magento\Integration\Api\CustomerTokenServiceInterface;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\GraphQlAbstract;

class AmGetGdprCustomerPersonalDataTest extends GraphQlAbstract
{
    public const MAIN_QUERY_KEY = 'amGetGdprCustomerPersonalData';
    public const CUSTOMER = 'new_customer@example.com';
    public const CUSTOMER_PASS = 'Qwert12345';

    /**
     * @var CustomerTokenServiceInterface
     */
    private $customerTokenService;

    protected function setUp(): void
    {
        $objectManager = Bootstrap::getObjectManager();
        $this->customerTokenService = $objectManager->get(CustomerTokenServiceInterface::class);
    }

    /**
     * @group amasty_gdpr
     *
     * @magentoDataFixture Magento/Customer/_files/new_customer.php
     */
    public function testRequestAmGdprAccountDeletion(): void
    {
        $fields = [
            'name' => 'customer_information',
            'personalDataPart' => [
                [
                    'label' => 'Name Prefix',
                    'value' => 'Mr.'
                ],
                [
                    'label' => 'First Name',
                    'value' => 'John'
                ],
                [
                    'label' => 'Middle Name/Initial',
                    'value' => 'A'
                ],
                [
                    'label' => 'Last Name',
                    'value' => 'Smith'
                ],
                [
                    'label' => 'Name Suffix',
                    'value' => 'Esq.'
                ],
                [
                    'label' => 'Email',
                    'value' => 'new_customer@example.com'
                ],
                [
                    'label' => 'Gender',
                    'value' => '1'
                ]
            ]
        ];

        $query = $this->getQuery(self::CUSTOMER_PASS);
        $response = $this->graphQlMutation($query, [], '', $this->getHeader());

        $this->assertArrayHasKey(self::MAIN_QUERY_KEY, $response);
        $this->assertGreaterThanOrEqual(15, $response[self::MAIN_QUERY_KEY]);
        $this->assertResponseFields($response[self::MAIN_QUERY_KEY][0], $fields);
        $this->assertEquals('customer_wishlist', end($response[self::MAIN_QUERY_KEY])['name']);
    }

    /**
     * Returns GraphQl query string
     */
    private function getQuery(string $password): string
    {
        return <<<MUTATION
mutation {
    amGetGdprCustomerPersonalData(
        input: {
            password: "$password"
    }) {
        name
        personalDataPart {
            label
            value
        }
    }
}
MUTATION;
    }

    /**
     * @param string $userName
     * @param string $password
     *
     * @return string[]
     * @throws AuthenticationException
     */
    private function getHeader(
        string $userName = self::CUSTOMER,
        string $password = self::CUSTOMER_PASS
    ): array {
        $customerToken = $this->customerTokenService->createCustomerAccessToken($userName, $password);
        return ['Authorization' => 'Bearer ' . $customerToken];
    }
}
