<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package GDPR GraphQl for Magento 2 (System)
 */

namespace Amasty\GdprGraphQl\Test\GraphQl;

use Amasty\Gdpr\Model\VisitorConsentLog\ResourceModel\VisitorConsentLog;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Framework\Exception\AuthenticationException;
use Magento\Integration\Api\CustomerTokenServiceInterface;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\GraphQlAbstract;

class AcceptAmGdprPolicyTest extends GraphQlAbstract
{
    public const MAIN_QUERY_KEY = 'acceptAmGdprPolicy';
    public const CUSTOMER = 'new_customer@example.com';
    public const CUSTOMER_PASS = 'Qwert12345';

    /**
     * @var CustomerRepositoryInterface
     */
    private $customerRepository;

    /**
     * @var VisitorConsentLog
     */
    private $visitorConsentLog;

    /**
     * @var CustomerTokenServiceInterface
     */
    private $customerTokenService;

    protected function setUp(): void
    {
        $objectManager = Bootstrap::getObjectManager();
        $this->customerRepository = $objectManager->create(CustomerRepositoryInterface::class);
        $this->visitorConsentLog = $objectManager->get(VisitorConsentLog::class);
        $this->customerTokenService = $objectManager->get(CustomerTokenServiceInterface::class);
    }

    /**
     * @group amasty_gdpr
     *
     * @magentoConfigFixture base_website amasty_gdpr/general/display_pp_popup 1
     *
     * @magentoDataFixture Magento/Customer/_files/new_customer.php
     *
     * @magentoApiDataFixture Amasty_GdprGraphQl::Test/GraphQl/_files/create_privacy_policy.php
     * @magentoApiDataFixture Amasty_GdprGraphQl::Test/GraphQl/_files/create_consent.php
     */
    public function testAcceptAmGdprPolicy(): void
    {
        $policyVersion = "v2.0";
        $fields = [
            'error' => false,
            'message' => "Privacy policy accepted."
        ];

        $query = $this->getQuery($policyVersion);
        $response = $this->graphQlMutation($query, [], '', $this->getHeader());

        $this->assertArrayHasKey(self::MAIN_QUERY_KEY, $response);
        $this->assertResponseFields($response[self::MAIN_QUERY_KEY], $fields);

        $customerId = $this->customerRepository->get(self::CUSTOMER)->getId();
        $this->assertEquals(
            $policyVersion,
            $this->visitorConsentLog->getPolicyVersionByCustomerId((int)$customerId, 1)
        );
    }

    /**
     * Returns GraphQl query string
     */
    private function getQuery(string $policy_version): string
    {
        return <<<MUTATION
mutation {
    acceptAmGdprPolicy(
        input: {
            policy_version: "$policy_version"
    }) {
        error
        message
    }
}
MUTATION;
    }

    /**
     * @param string $userName
     * @param string $password
     *
     * @return string[]
     * @throws AuthenticationException
     */
    private function getHeader(
        string $userName = self::CUSTOMER,
        string $password = self::CUSTOMER_PASS
    ): array {
        $customerToken = $this->customerTokenService->createCustomerAccessToken($userName, $password);
        return ['Authorization' => 'Bearer ' . $customerToken];
    }
}
