<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package GDPR GraphQl for Magento 2 (System)
 */

namespace Amasty\GdprGraphQl\Model\Resolver\Mutation;

use Amasty\Gdpr\Model\Consent\RegistryConstants;
use Magento\CustomerGraphQl\Model\Customer\GetCustomer;
use Magento\Framework\Event\ManagerInterface;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;

class GiveConsent implements ResolverInterface
{
    public const CONSENTS_KEY = 'consents';
    public const CONSENT_CODE_KEY = 'consent_code';
    public const CONSENT_FROM_KEY = 'consent_from';
    public const AGREED_KEY = 'has_agreement';

    /**
     * @var GetCustomer
     */
    private $getCustomer;

    /**
     * @var ManagerInterface
     */
    private $eventManager;

    public function __construct(
        GetCustomer $getCustomer,
        ManagerInterface $eventManager
    ) {
        $this->getCustomer = $getCustomer;
        $this->eventManager = $eventManager;
    }

    public function resolve(Field $field, $context, ResolveInfo $info, array $value = null, array $args = null)
    {
        $input = $args['input'] ?? [];
        $storeId = (int)$context->getExtensionAttributes()->getStore()->getId();

        if (!isset($input[self::CONSENTS_KEY]) || empty($input[self::CONSENTS_KEY])) {
            throw new GraphQlInputException(__('Consents must be specified.'));
        }

        $userConsentsData = $this->prepareUserConsents($input[self::CONSENTS_KEY], $storeId);

        try {
            $customerId = $this->getCustomer->execute($context)->getId();
        } catch (\Exception $e) {
            $customerId = 0;
        }

        foreach ($userConsentsData as $from => $consentData) {
            $this->eventManager->dispatch(
                'amasty_gdpr_consent_accept',
                [
                    //phpcs:ignore Magento2.Performance.ForeachArrayMerge.ForeachArrayMerge
                    RegistryConstants::CONSENTS => array_merge([], ...$consentData),
                    RegistryConstants::CONSENT_FROM => $from,
                    RegistryConstants::CUSTOMER_ID => $customerId,
                    RegistryConstants::STORE_ID => $storeId
                ]
            );
        }

        return [
            'error' => false,
            'message' => __('Changes has been saved')
        ];
    }

    private function prepareUserConsents(array $userConsents, int $storeId = 0): array
    {
        $preparedConsents = [];

        foreach ($userConsents as $consentData) {
            if (!isset($consentData[self::CONSENT_CODE_KEY]) || empty($consentData[self::CONSENT_CODE_KEY])) {
                throw new GraphQlInputException(__('Consent code must be specified.'));
            }

            if (!isset($consentData[self::CONSENT_FROM_KEY]) || empty($consentData[self::CONSENT_FROM_KEY])) {
                throw new GraphQlInputException(__('Consent from location must be specified.'));
            }

            if (!isset($consentData[self::AGREED_KEY])) {
                throw new GraphQlInputException(__('Agreement must be specified.'));
            }

            $code = $consentData[self::CONSENT_CODE_KEY];
            $from = $consentData[self::CONSENT_FROM_KEY];
            $agreement = $consentData[self::AGREED_KEY];

            $preparedConsents[$from][] = [$code => (bool)$agreement];
        }

        return $preparedConsents;
    }
}
