<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package GDPR GraphQl for Magento 2 (System)
 */

namespace Amasty\GdprGraphQl\Model\Resolver\Mutation\Customer;

use Amasty\Gdpr\Api\DeleteRequestRepositoryInterface;
use Amasty\Gdpr\Model\ActionLogger;
use Amasty\Gdpr\Model\Config;
use Amasty\Gdpr\Model\DeleteRequest;
use Amasty\Gdpr\Model\DeleteRequest\DeleteRequestSource;
use Amasty\Gdpr\Model\DeleteRequestFactory;
use Amasty\Gdpr\Model\Notification\NotificationsApplier;
use Amasty\Gdpr\Model\Notification\NotifiersProvider;
use Amasty\GdprGraphQl\Model\Resolver\CustomerProvider;
use Amasty\GdprGraphQl\Model\Resolver\Validation\CustomerHasActiveDeleteRequest;
use Amasty\GdprGraphQl\Model\Resolver\Validation\CustomerHasActiveGiftRegistry;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthenticationException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;

class Delete implements ResolverInterface
{
    /**
     * @var NotificationsApplier
     */
    private $notificationsApplier;

    /**
     * @var Config
     */
    private $configProvider;

    /**
     * @var ActionLogger
     */
    private $actionLogger;

    /**
     * @var CustomerProvider
     */
    private $customerProvider;

    /**
     * @var CustomerHasActiveGiftRegistry
     */
    private $customerHasActiveGiftRegistry;

    /**
     * @var CustomerHasActiveDeleteRequest
     */
    private $customerHasActiveDeleteRequest;

    /**
     * @var DeleteRequestFactory
     */
    private $deleteRequestFactory;

    /**
     * @var DeleteRequestRepositoryInterface
     */
    private $deleteRequestRepository;

    public function __construct(
        Config $configProvider,
        NotificationsApplier $notificationsApplier,
        ActionLogger $actionLogger,
        CustomerProvider $customerProvider,
        CustomerHasActiveGiftRegistry $customerHasActiveGiftRegistry,
        CustomerHasActiveDeleteRequest $customerHasActiveDeleteRequest,
        DeleteRequestFactory $deleteRequestFactory,
        DeleteRequestRepositoryInterface $deleteRequestRepository
    ) {
        $this->configProvider = $configProvider;
        $this->actionLogger = $actionLogger;
        $this->customerProvider = $customerProvider;
        $this->customerHasActiveGiftRegistry = $customerHasActiveGiftRegistry;
        $this->customerHasActiveDeleteRequest = $customerHasActiveDeleteRequest;
        $this->deleteRequestFactory = $deleteRequestFactory;
        $this->deleteRequestRepository = $deleteRequestRepository;
        $this->notificationsApplier = $notificationsApplier;
    }

    public function resolve(Field $field, $context, ResolveInfo $info, array $value = null, array $args = null)
    {
        if (!$this->configProvider->isAllowed(Config::DELETE)) {
            throw new GraphQlAuthenticationException(__('Access denied.'));
        }

        $customer = $this->customerProvider->getCustomer($context, $args ?? []);

        if ($this->customerHasActiveDeleteRequest->execute($customer)) {
            return [
                'error' => true,
                'message' => __('Your delete account request is awaiting for the review by the administrator.')
            ];
        }

        if ($this->customerHasActiveGiftRegistry->execute($customer)) {
            return [
                'error' => true,
                'message' => __('We can not process your account deletion request right now, ' .
                    'because you have active Gift Registry.')
            ];
        }

        try {
            /** @var DeleteRequest $request */
            $request = $this->deleteRequestFactory->create();
            $request->setCustomerId((int)$customer->getId());
            $request->setGotFrom(DeleteRequestSource::CUSTOMER_REQUEST);
            $this->deleteRequestRepository->save($request);
            $this->actionLogger->logAction('delete_request_submitted', $request->getCustomerId());
            $this->notificationsApplier->apply(
                NotifiersProvider::EVENT_DELETE_REQUEST_CREATED,
                (int)$customer->getId()
            );
        } catch (\Exception $e) {
            return [
                'error' => true,
                'message' => __('An error has occurred.')
            ];
        }

        return [
            'error' => false,
            'message' => __('Thank you, your account delete request was recorded.')
        ];
    }
}
