<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package GDPR GraphQl for Magento 2 (System)
 */

namespace Amasty\GdprGraphQl\Model\Resolver\Mutation\Customer;

use Amasty\Gdpr\Model\Anonymization\Anonymizer;
use Amasty\Gdpr\Model\Config;
use Amasty\GdprGraphQl\Model\Resolver\CustomerProvider;
use Amasty\GdprGraphQl\Model\Resolver\Validation\CustomerHasActiveGiftRegistry;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthenticationException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;

class Anonymise implements ResolverInterface
{
    /**
     * @var Config
     */
    private $configProvider;

    /**
     * @var Anonymizer
     */
    private $anonymizer;

    /**
     * @var CustomerProvider
     */
    private $customerProvider;

    /**
     * @var CustomerHasActiveGiftRegistry
     */
    private $customerHasActiveGiftRegistry;

    public function __construct(
        Config $configProvider,
        Anonymizer $anonymizer,
        CustomerProvider $customerProvider,
        CustomerHasActiveGiftRegistry $customerHasActiveGiftRegistry
    ) {
        $this->configProvider = $configProvider;
        $this->anonymizer = $anonymizer;
        $this->customerProvider = $customerProvider;
        $this->customerHasActiveGiftRegistry = $customerHasActiveGiftRegistry;
    }

    public function resolve(Field $field, $context, ResolveInfo $info, array $value = null, array $args = null)
    {
        if (!$this->configProvider->isAllowed(Config::ANONYMIZE)) {
            throw new GraphQlAuthenticationException(__('Access denied.'));
        }

        $customer = $this->customerProvider->getCustomer($context, $args ?? []);
        $activeOrdersIncrementIds = array_map(function ($orderData) {
            return $orderData['increment_id'] ?? '';
        }, $this->anonymizer->getCustomerActiveOrders((int)$customer->getId()));

        if (count($activeOrdersIncrementIds) > 0) {
            return [
                'error' => true,
                'message' => __(
                    'We can not anonymize your account right now, because you have non-completed order(s):%1',
                    implode(' ', $activeOrdersIncrementIds)
                )
            ];
        }

        if ($this->customerHasActiveGiftRegistry->execute($customer)) {
            return [
                'error' => true,
                'message' => __('We can not anonymize your account right now, because you have active Gift Registry.')
            ];
        }

        try {
            $this->anonymizer->anonymizeCustomer((int)$customer->getId());
        } catch (\Exception $e) {
            return [
                'error' => true,
                'message' => __('An error has occurred.')
            ];
        }

        return [
            'error' => false,
            'message' => __('Anonymisation was successful.')
        ];
    }
}
