<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package GDPR GraphQl for Magento 2 (System)
 */

namespace Amasty\GdprGraphQl\Model\Resolver\Mutation;

use Amasty\Gdpr\Model\Config;
use Amasty\Gdpr\Model\ConsentLogger;
use Amasty\Gdpr\Model\ConsentProvider;
use Amasty\Gdpr\Model\ConsentVisitorLogger;
use Amasty\Gdpr\Model\Source\ConsentLinkType;
use Magento\Customer\Model\Session;
use Magento\CustomerGraphQl\Model\Customer\GetCustomer;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Store\Model\StoreManagerInterface;
use Psr\Log\LoggerInterface;

class AcceptPolicy implements ResolverInterface
{
    public const POLICY_VERSION_KEY = 'policy_version';

    /**
     * @var Config
     */
    private $configProvider;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var Session
     */
    private $customerSession;

    /**
     * @var ConsentProvider
     */
    private $consentProvider;

    /**
     * @var ConsentVisitorLogger
     */
    private $consentVisitorLogger;

    /**
     * @var ConsentLogger
     */
    private $consentLogger;

    /**
     * @var GetCustomer
     */
    private $getCustomer;

    /**
     * @var LoggerInterface
     */
    private $logger;

    public function __construct(
        Config $configProvider,
        StoreManagerInterface $storeManager,
        Session $customerSession,
        ConsentProvider $consentProvider,
        ConsentVisitorLogger $consentVisitorLogger,
        ConsentLogger $consentLogger,
        GetCustomer $getCustomer,
        LoggerInterface $logger
    ) {
        $this->configProvider = $configProvider;
        $this->storeManager = $storeManager;
        $this->customerSession = $customerSession;
        $this->consentProvider = $consentProvider;
        $this->consentVisitorLogger = $consentVisitorLogger;
        $this->consentLogger = $consentLogger;
        $this->getCustomer = $getCustomer;
        $this->logger = $logger;
    }

    public function resolve(Field $field, $context, ResolveInfo $info, array $value = null, array $args = null)
    {
        $input = $args['input'] ?? [];

        if (!$this->configProvider->isModuleEnabled() || !$this->configProvider->isDisplayPpPopup()) {
            return [];
        }

        if (!isset($input[self::POLICY_VERSION_KEY]) || empty($input[self::POLICY_VERSION_KEY])) {
            throw new GraphQlInputException(__('Invalid Policy version.'));
        }

        $error = true;
        $policyVersion = $input[self::POLICY_VERSION_KEY];

        try {
            $email = null;
            if (!$this->customerSession->getCustomerId()) {
                try {
                    $customerId = (int)$this->getCustomer->execute($context)->getId();
                } catch (\Exception $e) {
                    $customerId = 0;
                }
            } else {
                $customerId = (int)$this->customerSession->getCustomerId();
            }
            $sessionId = (string)$this->customerSession->getSessionId();
            $storeId = (int)$this->storeManager->getStore()->getId();
            if ($customerData = $this->customerSession->getCustomerData()) {
                $email = $customerData->getEmail();
            }
            $this->consentVisitorLogger->log($policyVersion, $customerId, $sessionId);
            $this->consentLogger->logParams(
                $customerId,
                ConsentLogger::PRIVACY_POLICY_POPUP,
                ConsentLinkType::PRIVACY_POLICY,
                true,
                null,
                $email
            );

            /** @var \Amasty\Gdpr\Model\Consent\Consent $consent */
            foreach ($this->consentProvider->getConsentsByStore($storeId) as $consent) {
                $this->consentLogger->logParams(
                    $customerId,
                    ConsentLogger::PRIVACY_POLICY_POPUP,
                    $consent->getPrivacyLinkType(),
                    true,
                    $consent->getConsentCode(),
                    $email
                );
            }
            $message = __('Privacy policy accepted.');
            $error = false;
        } catch (\Exception $e) {
            $message = __('An error has occurred');
            $this->logger->critical($e);
        }

        return [
            'error' => $error,
            'message' => $message
        ];
    }
}
