<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package GDPR GraphQl for Magento 2 (System)
 */

namespace Amasty\GdprGraphQl\Model\Resolver;

use Amasty\Gdpr\Api\Data\WithConsentInterface;
use Amasty\Gdpr\Model\Consent\Consent;
use Amasty\Gdpr\Model\PolicyRepository;
use Amasty\Gdpr\Model\ResourceModel\WithConsent\CollectionFactory as WithConsentCollectionFactory;
use Amasty\GdprGraphQl\Model\ConsentsDataProvider;
use Magento\CustomerGraphQl\Model\Customer\GetCustomer;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Query\Resolver\ContextInterface;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;

class GetConsent implements ResolverInterface
{
    /**
     * @var GetCustomer
     */
    private $getCustomer;

    /**
     * @var WithConsentCollectionFactory
     */
    private $withConsentCollectionFactory;

    /**
     * @var PolicyRepository
     */
    private $policyRepository;

    /**
     * @var ConsentsDataProvider
     */
    private $consentsDataProvider;

    public function __construct(
        GetCustomer $getCustomer,
        WithConsentCollectionFactory $withConsentCollectionFactory,
        PolicyRepository $policyRepository,
        ConsentsDataProvider $consentsDataProvider
    ) {
        $this->getCustomer = $getCustomer;
        $this->withConsentCollectionFactory = $withConsentCollectionFactory;
        $this->policyRepository = $policyRepository;
        $this->consentsDataProvider = $consentsDataProvider;
    }

    public function resolve(Field $field, $context, ResolveInfo $info, array $value = null, array $args = null)
    {
        $storeId = (int)$context->getExtensionAttributes()->getStore()->getId();
        $consents = $this->consentsDataProvider->getData($args['location'] ?? null, $storeId);
        $agreedConsentCodes = $this->getAgreedConsentCodes($context, $storeId);

        return array_map(function (Consent $consent) use ($agreedConsentCodes) {
            $consent->setData('has_agreement', in_array($consent->getConsentCode(), $agreedConsentCodes));

            return $consent;
        }, $consents);
    }

    private function getAgreedConsentCodes(ContextInterface $context, int $storeId): array
    {
        $currentPolicy = $this->policyRepository->getCurrentPolicy($storeId);

        try {
            $customerId = (int)$this->getCustomer->execute($context)->getId();
            $consentLogCollection = $this->withConsentCollectionFactory->create();
            $consentLogCollection->filterByLastConsentRecord()
                ->filterByPolicyVersionAndLinkType($currentPolicy->getPolicyVersion(), $storeId)
                ->addFieldToFilter('main_table.' . WithConsentInterface::CUSTOMER_ID, $customerId)
                ->addFieldToFilter('main_table.' . WithConsentInterface::ACTION, true);

            $agreedConsentCodes = $consentLogCollection->getColumnValues(
                WithConsentInterface::CONSENT_CODE
            );
        } catch (\Exception $e) {
            $agreedConsentCodes = [];
        }

        return $agreedConsentCodes;
    }
}
