<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Google Analytics GA4 by Amasty
 */

namespace Amasty\GA4\Model\Cookie;

use Amasty\Base\Model\ModuleInfoProvider;
use Amasty\GA4\Model\ThirdParty\ModuleChecker;
use Amasty\GdprCookie\Model\CookieGroup;
use Amasty\GdprCookie\Model\ResourceModel\CookieGroup\Collection;
use Magento\Framework\ObjectManagerInterface;

class GdprCookieAdapter
{
    /**
     * @var ObjectManagerInterface
     */
    private ObjectManagerInterface $objectManager;

    /**
     * @var bool
     */
    private bool $isAllowed;

    public function __construct(
        ModuleChecker $moduleChecker,
        ObjectManagerInterface $objectManager,
        ModuleInfoProvider $moduleInfoProvider
    ) {
        $this->isAllowed = $this->checkAvailability($moduleChecker, $moduleInfoProvider);
        $this->objectManager = $objectManager;
    }

    public function isModuleValid(): bool
    {
        return $this->isAllowed;
    }

    public function getCookieGroups(): array
    {
        if (!$this->isAllowed) {
            return [];
        }
        $cookieGroups = [];
        $cookieGroupsCollection = $this->objectManager->create(Collection::class);

        /** @var CookieGroup $cookieGroup */
        foreach ($cookieGroupsCollection->getItems() as $cookieGroup) {
            $cookieGroups[$cookieGroup->getId()] = $cookieGroup->getName();
        }

        return $cookieGroups;
    }

    private function checkAvailability(ModuleChecker $moduleChecker, ModuleInfoProvider $moduleInfoProvider): bool
    {
        if (!$moduleChecker->isGdprCookieEnabled()) {
            return false;
        }

        $requiredVersion = '2.10.1';
        $originalVersion = $moduleInfoProvider->getModuleInfo(ModuleChecker::COOKIE_MODULE)['version'] ?? '0.0.1';

        return version_compare($originalVersion, $requiredVersion) >= 0;
    }
}
