<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Free Gift Hyva Compatibility
 */
declare(strict_types=1);

namespace Amasty\PromoHyvaCompatibility\ViewModel;

use Magento\Framework\App\ProductMetadata;
use Magento\Framework\App\ProductMetadataInterface;
use Magento\Framework\View\Element\Block\ArgumentInterface;

class GraphQlQueries implements ArgumentInterface
{
    /**
     * @var ProductMetadataInterface
     */
    private ProductMetadataInterface $productMetadata;

    /**
     * Constructor
     *
     * @param ProductMetadataInterface $productMetadata
     */
    public function __construct(
        ProductMetadataInterface $productMetadata
    ) {
        $this->productMetadata = $productMetadata;
    }

    /**
     * Graphql query with get products
     *
     * @return string
     */
    public function getProduct(): string
    {
        return "promoProducts(filter: { sku: { in: \$skuArray } }) {
            items {
                __typename
                id
                uid
                sku
                name
                stock_status
                url_key
                url_suffix
                small_image {
                    url
                }
                # eslint-disable-next-line @graphql-eslint/require-id-when-available
                media_gallery_entries {
                    uid
                    label
                    position
                    disabled
                    file
                }
                price_range {
                    minimum_price {
                        regular_price {
                            value
                            currency
                        }
                        final_price {
                            currency
                            value
                        }
                    }
                    maximum_price {
                        regular_price {
                            value
                            currency
                        }
                        final_price {
                            value
                            currency
                        }
                    }
                }
                ... on CustomizableProductInterface {
                    options {
                        title
                        required
                        sort_order
                        option_id
                        uid
                        ... on CustomizableFieldOption {
                            product_sku
                            field_option: value {
                                uid
                                sku
                                price
                                price_type
                                max_characters
                            }
                        }
                        ... on CustomizableAreaOption {
                            product_sku
                            area_option: value {
                                uid
                                sku
                                price
                                price_type
                                max_characters
                            }
                        }
                        ... on CustomizableDateOption {
                            product_sku
                            date_option: value {
                                uid
                                sku
                                price
                                price_type
                            }
                        }
                        ... on CustomizableDropDownOption {
                            drop_down_option: value {
                                uid
                                option_type_id
                                sku
                                price
                                price_type
                                title
                                sort_order
                            }
                        }
                        ... on CustomizableRadioOption {
                            radio_option: value {
                                uid
                                option_type_id
                                sku
                                price
                                price_type
                                title
                                sort_order
                            }
                        }
                        ... on CustomizableCheckboxOption {
                            checkbox_option: value {
                                uid
                                option_type_id
                                sku
                                price
                                price_type
                                title
                                sort_order
                            }
                        }
                        ... on CustomizableMultipleOption {
                            multiple_option: value {
                                uid
                                option_type_id
                                sku
                                price
                                price_type
                                title
                                sort_order
                            }
                        }
                        ... on CustomizableFileOption {
                            product_sku
                            file_option: value {
                                uid
                                sku
                                price
                                price_type
                                file_extension
                                image_size_x
                                image_size_y
                            }
                        }
                    }
                }
                ... on BundleProduct {
                    # eslint-disable-next-line @graphql-eslint/require-id-when-available
                    dynamic_sku
                    dynamic_price
                    dynamic_weight
                    price_view
                    ship_bundle_items
                    items {
                        option_id
                        title
                        required
                        type
                        position
                        sku
                        options {
                            id
                            uid
                            quantity
                            position
                            is_default
                            price
                            price_type
                            can_change_quantity
                            label
                            product {
                                id
                                name
                                sku
                                type_id
                                price_range {
                                    maximum_price {
                                        final_price {
                                            currency
                                            value
                                        }
                                    }
                                    minimum_price {
                                        final_price {
                                            currency
                                            value
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
                {$this->getGiftCardProductQuery()}
                ... on ConfigurableProduct {
                    # eslint-disable-next-line @graphql-eslint/require-id-when-available
                    configurable_options {
                        attribute_code
                        attribute_id
                        uid
                        label
                        # eslint-disable-next-line @graphql-eslint/require-id-when-available
                        values {
                            uid
                            default_label
                            label
                            store_label
                            use_default_value
                            value_index
                            swatch_data {
                                ... on ImageSwatchData {
                                    thumbnail
                                }
                                value
                            }
                        }
                    }
                    variants {
                        attributes {
                            code
                            value_index
                        }
                        # eslint-disable-next-line @graphql-eslint/require-id-when-available
                        product {
                            uid
                            # eslint-disable-next-line @graphql-eslint/require-id-when-available
                            media_gallery_entries {
                                uid
                                disabled
                                file
                                label
                                position
                            }
                            sku
                            stock_status
                            price_range {
                                minimum_price {
                                    final_price {
                                        currency
                                        value
                                    }
                                }
                            }
                            price {
                                regularPrice {
                                    amount {
                                        currency
                                        value
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }";
    }

    /**
     * Get Graphql mutation add free gift product
     *
     * @return string
     */
    public function getAddFreeGiftsToCart(): string
    {
        return 'amPromoAddFreeGiftsToCart(cartId: $cartId, promoItems: $promoItems) {
                    cart {
                        total_quantity
                    }
                    user_errors
                }';
    }

    /**
     * Get Graphql query with available for ampromo
     *
     * @return string
     */
    public function getAvailableFreeGifts(): string
    {
        return 'amPromoGetAvailableFreeGifts(cartId: $cartId) {
                    available_gifts_qty
                    promo_rule_type(cartId: $cartId)
                    promo_items {
                        available_qty
                        discount(cartId: $cartId) {
                            discount_item
                            minimal_price
                        }
                        price
                        sku
                    }
                }';
    }

    /**
     * Get Graphql query with store config for ampromo
     *
     * @return string
     */
    public function getStoreConfig(): string
    {
        return 'storeConfig {
            ampromo_messages_display_remaining_gifts_counter
            ampromo_messages_gift_selection_method
            ampromo_messages_show_price_in_popup
            ampromo_messages_auto_open_popup
            ampromo_messages_add_message
            ampromo_messages_add_button_title
            ampromo_messages_popup_title
        }';
    }

    /**
     * Check if gift card is available
     *
     * @return bool
     */
    public function isGiftCardAvailable(): bool
    {
        return $this->productMetadata->getEdition() !== ProductMetadata::EDITION_NAME;
    }

    /**
     * Check if gift card is available return query
     *
     * @return string
     */
    private function getGiftCardProductQuery(): string
    {
        if (!$this->isGiftCardAvailable()) {
            return '';
        }

        return "...on GiftCardProduct {
                      giftcard_type
                      open_amount_max
                      open_amount_min
                      allow_open_amount
                      allow_message
                      lifetime
                      message_max_length
                      gift_card_options {
                        title
                        __typename
                        required
                        ... on CustomizableFieldOption {
                          value {
                            uid
                          }
                        }
                      },
                      giftcard_amounts {
                        value
                        uid
                        attribute_id
                      }
                  }";
    }
}
