<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Full Page Cache Warmer for Magento 2
 */

namespace Amasty\Fpc\Model;

use Amasty\Fpc\Api\QueuePageRepositoryInterface;
use Amasty\Fpc\Exception\LockException;
use Amasty\Fpc\Model\Config\Source\QuerySource;
use Amasty\Fpc\Model\Crawler\Crawler;
use Amasty\Fpc\Model\Queue\ProcessMetaInfo;
use Amasty\Fpc\Model\ResourceModel\Activity;
use Amasty\Fpc\Model\ResourceModel\Queue\Page\Collection as PageCollection;
use Amasty\Fpc\Model\ResourceModel\Queue\Page\CollectionFactory as PageCollectionFactory;
use Amasty\Fpc\Model\Source\PagesProvider;

class Queue
{
    /**
     * @var Config
     */
    private $config;

    /**
     * @var PageCollectionFactory
     */
    private $pageCollectionFactory;

    /**
     * @var QueuePageRepository
     */
    private $pageRepository;

    /**
     * @var Crawler
     */
    private $crawler;

    /**
     * @var PagesProvider
     */
    private $pagesProvider;

    /**
     * @var ProcessMetaInfo
     */
    private $processMetaInfo;

    public function __construct(
        Config $config,
        PageCollectionFactory $pageCollectionFactory,
        QueuePageRepositoryInterface $pageRepository,
        Crawler $crawler,
        PagesProvider $pagesProvider,
        ProcessMetaInfo $processMetaInfo
    ) {
        $this->config = $config;
        $this->pageCollectionFactory = $pageCollectionFactory;
        $this->pageRepository = $pageRepository;
        $this->crawler = $crawler;
        $this->pagesProvider = $pagesProvider;
        $this->processMetaInfo = $processMetaInfo;
    }

    protected function lock()
    {
        if ($this->processMetaInfo->isQueueLocked()) {
            throw new LockException(__('Another lock detected (the Warmer queue is in a progress).'));
        }

        $this->processMetaInfo->setIsQueueLocked(true);
    }

    protected function unlock()
    {
        $this->processMetaInfo->setIsQueueLocked(false);
    }

    public function forceUnlock()
    {
        $this->processMetaInfo->setIsQueueLocked(false);
    }

    public function generate(): array
    {
        $this->lock();
        $processedItems = 0;
        $queueLimit = $this->config->getQueueLimit();
        $sourceType = $this->config->getSourceType();
        $sourcePages = $this->pagesProvider->getSourcePages($sourceType, $queueLimit);

        if (empty($sourcePages)) {
            $this->unlock();

            return [false, $processedItems];
        }

        try {
            $this->pageRepository->clear();
        } catch (\Exception $e) {
            $this->unlock();

            return [false, $processedItems];
        }

        foreach ($sourcePages as $page) {
            $this->pageRepository->addPage($page);
            $processedItems++;

            if (!$this->processMetaInfo->isQueueLocked()) {
                return [false, $processedItems];
            }
        }

        $this->unlock();
        $this->processMetaInfo->setTotalPagesQueued($processedItems);
        $this->processMetaInfo->resetTotalPagesCrawled();

        return [true, $processedItems];
    }

    public function process(): int
    {
        $this->lock();
        $uncachedPagesCollection = $this->getUncachedPages();
        $this->crawler->processPages($uncachedPagesCollection);
        $this->processMetaInfo->addToTotalPagesCrawled($uncachedPagesCollection->count());

        if ((int)$this->config->getSourceType() === QuerySource::SOURCE_ACTIVITY
            && $this->config->isUseVisitParams()
        ) {
            $uncachedPagesCollection = $this->getUncachedActivityPages();
            $this->crawler->processUsingVisitParams($uncachedPagesCollection);
            $this->processMetaInfo->addToTotalPagesCrawled($uncachedPagesCollection->count());
        }

        $this->unlock();

        return $uncachedPagesCollection->count();
    }

    private function getUncachedPages()
    {
        $pageCollection = $this->pageCollectionFactory->create()->setOrder('rate');

        if ((int)$this->config->getSourceType() === QuerySource::SOURCE_ACTIVITY
            && $this->config->isUseVisitParams()
        ) {
            $pageCollection->addFieldToFilter('activity_id', ['null' => true]);
        }

        $pageCollection->setPageSize($this->config->getBatchSize());

        return $pageCollection;
    }

    private function getUncachedActivityPages(): PageCollection
    {
        $pageCollection = $this->pageCollectionFactory->create()->setOrder('main_table.rate');
        $pageCollection->getSelect()->joinInner(
            ['activity_data' => $pageCollection->getTable(Activity::TABLE_NAME)],
            'main_table.activity_id = activity_data.id',
            [
                'activity_data.mobile',
                'activity_data.currency',
                'activity_data.customer_group'
            ]
        );
        $pageCollection->setPageSize($this->config->getBatchSize());

        return $pageCollection;
    }
}
