<?php

declare(strict_types=1);

namespace Amasty\FacebookPixelPro\Model;

use Amasty\FacebookPixelPro\Api\Data\ServerSideSettingsInterface;
use Amasty\FacebookPixelPro\Api\Data\ServerSideSettingsInterfaceFactory;
use Amasty\FacebookPixelPro\Api\ServerSideSettingsRepositoryInterface;
use Amasty\FacebookPixelPro\Model\ResourceModel\ServerSideSettings as ServerSideSettingsResource;
use Amasty\FacebookPixelPro\Model\ResourceModel\ServerSideSettings\CollectionFactory
    as ServerSideSettingsCollectionFactory;
use Exception;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Api\SearchResultsInterface;
use Magento\Framework\Api\SearchResultsInterfaceFactory;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;

class ServerSideSettingsRepository implements ServerSideSettingsRepositoryInterface
{
    /**
     * @var ServerSideSettingsResource
     */
    private $resource;

    /**
     * @var ServerSideSettingsInterfaceFactory
     */
    private $modelFactory;

    /**
     * @var ServerSideSettingsCollectionFactory
     */
    private $collectionFactory;

    /**
     * @var CollectionProcessorInterface
     */
    private $collectionProcessor;

    /**
     * @var SearchResultsInterfaceFactory
     */
    private $searchResultsFactory;

    public function __construct(
        ServerSideSettingsResource $resource,
        ServerSideSettingsInterfaceFactory $modelFactory,
        ServerSideSettingsCollectionFactory $collectionFactory,
        CollectionProcessorInterface $collectionProcessor,
        SearchResultsInterfaceFactory $searchResultsFactory
    ) {
        $this->resource = $resource;
        $this->modelFactory = $modelFactory;
        $this->collectionFactory = $collectionFactory;
        $this->collectionProcessor = $collectionProcessor;
        $this->searchResultsFactory = $searchResultsFactory;
    }

    /**
     * @param ServerSideSettingsInterface $pixel
     * @return ServerSideSettingsInterface
     * @throws CouldNotSaveException
     */
    public function save(ServerSideSettingsInterface $pixel): ServerSideSettingsInterface
    {
        try {
            $this->resource->save($pixel);
        } catch (Exception $exception) {
            throw new CouldNotSaveException(__($exception->getMessage()));
        }

        return $pixel;
    }

    /**
     * @param int $entityId
     * @return ServerSideSettingsInterface
     * @throws NoSuchEntityException
     */
    public function getById(int $entityId): ServerSideSettingsInterface
    {
        $pixel = $this->modelFactory->create();
        $this->resource->load($pixel, $entityId);

        if (!$pixel->getPixelId()) {
            throw new NoSuchEntityException(__('Pixel with id "%1" does not exist.', $entityId));
        }

        return $pixel;
    }

    /**
     * @param int $pixelEntityId
     * @return ServerSideSettingsInterface
     * @throws NoSuchEntityException
     */
    public function getByPixelEntityId(int $pixelEntityId): ServerSideSettingsInterface
    {
        $model = $this->modelFactory->create();
        $this->resource->load($model, $pixelEntityId, ServerSideSettingsInterface::PIXEL_ENTITY_ID);

        if (!$model->getId()) {
            throw new NoSuchEntityException(
                __('Server Side Settings with pixel_entity_id id "%1" does not exist.', $pixelEntityId)
            );
        }

        return $model;
    }

    /**
     * @param int $entityId
     * @return ServerSideSettingsInterface|null
     */
    public function findById(int $entityId): ?ServerSideSettingsInterface
    {
        $pixel = $this->modelFactory->create();
        $this->resource->load($pixel, $entityId);

        if (!$pixel->getPixelId()) {
            return null;
        }

        return $pixel;
    }

    /**
     * @param SearchCriteriaInterface $searchCriteria
     * @return SearchResultsInterface
     */
    public function getList(SearchCriteriaInterface $searchCriteria): SearchResultsInterface
    {
        $collection = $this->collectionFactory->create();

        $this->collectionProcessor->process($searchCriteria, $collection);
        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($searchCriteria);
        $searchResults->setItems($collection->getItems());
        $searchResults->setTotalCount($collection->getSize());

        return $searchResults;
    }

    /**
     * @param ServerSideSettingsInterface $pixel
     * @return bool
     * @throws CouldNotDeleteException
     */
    public function delete(ServerSideSettingsInterface $pixel): bool
    {
        try {
            $this->resource->delete($pixel);
        } catch (Exception $exception) {
            throw new CouldNotDeleteException(__($exception->getMessage()));
        }

        return true;
    }

    /**
     * @param int $entityId
     * @return bool
     * @throws CouldNotDeleteException
     * @throws NoSuchEntityException
     */
    public function deleteById(int $entityId): bool
    {
        return $this->delete($this->getById($entityId));
    }
}
