<?php

declare(strict_types=1);

namespace Amasty\FacebookPixelPro\Model;

use Amasty\FacebookPixelPro\Api\Data\EventQueueInterface;
use Amasty\FacebookPixelPro\Api\Data\EventQueueInterfaceFactory;
use Amasty\FacebookPixelPro\Api\EventQueueRepositoryInterface;
use Amasty\FacebookPixelPro\Model\Api\ApiCaller;
use Magento\Framework\Serialize\SerializerInterface;
use Psr\Log\LoggerInterface;

class RequestProcessor
{
    /**
     * @var ApiCaller
     */
    private $apiCaller;

    /**
     * @var EventQueueRepositoryInterface
     */
    private $eventQueueRepository;

    /**
     * @var EventQueueInterfaceFactory
     */
    private $eventQueueFactory;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    public function __construct(
        ApiCaller $apiCaller,
        EventQueueRepositoryInterface $eventQueueRepository,
        EventQueueInterfaceFactory $eventQueueFactory,
        LoggerInterface $logger,
        SerializerInterface $serializer,
        ConfigProvider $configProvider
    ) {
        $this->apiCaller = $apiCaller;
        $this->eventQueueRepository = $eventQueueRepository;
        $this->eventQueueFactory = $eventQueueFactory;
        $this->logger = $logger;
        $this->serializer = $serializer;
        $this->configProvider = $configProvider;
    }

    /**
     * @param string $pixelId
     * @param string $token
     * @param array $data
     * @param string|null $testEventCode
     */
    public function processRequest(string $pixelId, string $token, array $data, ?string $testEventCode = null): void
    {
        try {
            if ($this->configProvider->isDelayedDispatchEnabled()) {
                /** @var EventQueueInterface $eventQueue */
                $eventQueue = $this->eventQueueFactory->create();
                $eventQueue->setPixelId($pixelId);
                $eventQueue->setToken($token);
                $eventQueue->setEventData($this->serializer->serialize($data));
                $eventQueue->setTestEventCode($testEventCode);

                $this->eventQueueRepository->save($eventQueue);

                return;
            }

            $this->sendRequest($pixelId, $token, [$data], $testEventCode);
        } catch (\Exception $e) {
            $this->logger->critical($e);
        }
    }

    /**
     * @param string $pixelId
     * @param string $token
     * @param array $data
     * @param string|null $testEventCode
     */
    public function sendRequest(string $pixelId, string $token, array $data, ?string $testEventCode = null): void
    {
        $this->apiCaller->sendEvent($pixelId, $token, $data, $testEventCode);
    }
}
