<?php

declare(strict_types=1);

namespace Amasty\FacebookPixelPro\Model;

use Amasty\FacebookPixelPro\Api\Data\EventQueueInterface;
use Amasty\FacebookPixelPro\Api\Data\EventQueueInterfaceFactory;
use Amasty\FacebookPixelPro\Api\EventQueueRepositoryInterface;
use Amasty\FacebookPixelPro\Model\ResourceModel\EventQueue as EventQueueResource;
use Amasty\FacebookPixelPro\Model\ResourceModel\EventQueue\CollectionFactory as EventQueueCollectionFactory;
use Exception;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Api\SearchResultsInterface;
use Magento\Framework\Api\SearchResultsInterfaceFactory;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;

class EventQueueRepository implements EventQueueRepositoryInterface
{
    /**
     * @var EventQueueResource
     */
    private $resource;

    /**
     * @var EventQueueInterfaceFactory
     */
    private $modelFactory;

    /**
     * @var EventQueueCollectionFactory
     */
    private $collectionFactory;

    /**
     * @var CollectionProcessorInterface
     */
    private $collectionProcessor;

    /**
     * @var SearchResultsInterfaceFactory
     */
    private $searchResultsFactory;

    public function __construct(
        EventQueueResource $resource,
        EventQueueInterfaceFactory $modelFactory,
        EventQueueCollectionFactory $collectionFactory,
        CollectionProcessorInterface $collectionProcessor,
        SearchResultsInterfaceFactory $searchResultsFactory
    ) {
        $this->resource = $resource;
        $this->modelFactory = $modelFactory;
        $this->collectionFactory = $collectionFactory;
        $this->collectionProcessor = $collectionProcessor;
        $this->searchResultsFactory = $searchResultsFactory;
    }

    /**
     * @param EventQueueInterface $eventQueue
     * @return EventQueueInterface
     * @throws CouldNotSaveException
     */
    public function save(EventQueueInterface $eventQueue): EventQueueInterface
    {
        try {
            $this->resource->save($eventQueue);
        } catch (Exception $exception) {
            throw new CouldNotSaveException(__($exception->getMessage()));
        }

        return $eventQueue;
    }

    /**
     * @param int $entityId
     * @return EventQueueInterface
     * @throws NoSuchEntityException
     */
    public function getById(int $entityId): EventQueueInterface
    {
        $eventQueue = $this->modelFactory->create();
        $this->resource->load($eventQueue, $entityId);

        if (!$eventQueue->getPixelId()) {
            throw new NoSuchEntityException(__('Event Queue with id "%1" does not exist.', $entityId));
        }

        return $eventQueue;
    }

    /**
     * @param int $entityId
     * @return EventQueueInterface|null
     */
    public function findById(int $entityId): ?EventQueueInterface
    {
        $eventQueue = $this->modelFactory->create();
        $this->resource->load($eventQueue, $entityId);

        if (!$eventQueue->getPixelId()) {
            return null;
        }

        return $eventQueue;
    }

    /**
     * @param SearchCriteriaInterface $searchCriteria
     * @return SearchResultsInterface
     */
    public function getList(SearchCriteriaInterface $searchCriteria): SearchResultsInterface
    {
        $collection = $this->collectionFactory->create();

        $this->collectionProcessor->process($searchCriteria, $collection);
        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($searchCriteria);
        $searchResults->setItems($collection->getItems());
        $searchResults->setTotalCount($collection->getSize());

        return $searchResults;
    }

    /**
     * @param EventQueueInterface $eventQueue
     * @return bool
     * @throws CouldNotDeleteException
     */
    public function delete(EventQueueInterface $eventQueue): bool
    {
        try {
            $this->resource->delete($eventQueue);
        } catch (Exception $exception) {
            throw new CouldNotDeleteException(__($exception->getMessage()));
        }

        return true;
    }

    /**
     * @param int $entityId
     * @return bool
     * @throws CouldNotDeleteException
     * @throws NoSuchEntityException
     */
    public function deleteById(int $entityId): bool
    {
        return $this->delete($this->getById($entityId));
    }
}
