<?php

declare(strict_types=1);

namespace Amasty\FacebookPixelPro\Model;

use Amasty\FacebookPixel\Api\Data\PixelInterface;
use Amasty\FacebookPixel\Model\PixelProvider;
use Amasty\FacebookPixelPro\Api\Data\ServerSideSettingsInterface;
use Amasty\FacebookPixelPro\Model\EventData\EventDataGenerator\EventDataGeneratorInterface;
use Amasty\FacebookPixelPro\Model\EventData\EventDataGeneratorPool;
use Magento\Framework\Exception\LocalizedException;
use Psr\Log\LoggerInterface;

class EventProcessor
{
    /**
     * @var EventDataGeneratorPool
     */
    private $eventDataGeneratorPool;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var PixelProvider
     */
    private $pixelProvider;

    /**
     * @var UserInfoProvider
     */
    private $userInfoProvider;

    /**
     * @var RequestProcessor
     */
    private $requestProcessor;

    public function __construct(
        EventDataGeneratorPool $eventDataGeneratorPool,
        PixelProvider $pixelProvider,
        UserInfoProvider $userInfoProvider,
        LoggerInterface $logger,
        RequestProcessor $requestProcessor
    ) {
        $this->eventDataGeneratorPool = $eventDataGeneratorPool;
        $this->logger = $logger;
        $this->pixelProvider = $pixelProvider;
        $this->userInfoProvider = $userInfoProvider;
        $this->requestProcessor = $requestProcessor;
    }

    /**
     * @param string $eventCode
     * @param int $eventId
     * @param null $data
     * @param null $eventData
     * @throws LocalizedException
     */
    public function processEvent(string $eventCode, int $eventId, $data = null, $eventData = null): void
    {
        $eventDataGenerator = $this->eventDataGeneratorPool->getDataGenerator($eventCode);
        $pixels = $this->pixelProvider->getEnabledPixelsForCurrentStore();
        $eventTime = time();
        $userInfo = null;

        /** @var PixelInterface $pixel */
        foreach ($pixels as $pixel) {
            if (!$this->canProcessForPixel($pixel, $eventDataGenerator)) {
                continue;
            }

            if ($eventData === null) {
                try {
                    $eventData = $eventDataGenerator->getEventData($data);
                } catch (\Exception $e) {
                    $this->logger->critical($e);

                    return;
                }
            }

            $fullEventData = [
                'event_name' => $eventDataGenerator->getEventName(),
                'custom_data' => $eventData,
                'event_time' => $eventTime,
                'event_id' => $eventId,
                'action_source' => 'website',
            ];

            if ($userInfo === null) {
                $userInfo = $this->userInfoProvider->getUserInfo();
            }

            // phpcs:ignore Magento2.Performance.ForeachArrayMerge.ForeachArrayMerge
            $fullEventData = array_merge($fullEventData, ['user_data' => $userInfo]);
            /** @var ServerSideSettingsInterface $serverSideData */
            $serverSideData = $pixel->getExtensionAttributes()->getServerSideData();

            $this->requestProcessor->processRequest(
                $pixel->getPixelId(),
                $serverSideData->getToken(),
                $fullEventData,
                $serverSideData->getTestEventCode()
            );
        }
    }

    /**
     * @param PixelInterface $pixel
     * @param EventDataGeneratorInterface $eventDataGenerator
     * @return bool
     */
    private function canProcessForPixel(PixelInterface $pixel, EventDataGeneratorInterface $eventDataGenerator): bool
    {
        /** @var ServerSideSettingsInterface $serverSideSettings */
        $serverSideSettings = $pixel->getExtensionAttributes()->getServerSideData();
        if (!$serverSideSettings
            || !$serverSideSettings->getIsActive()
            || !$serverSideSettings->getToken()
            || !$eventDataGenerator->isEventEnabledForPixel($pixel)) {
            return false;
        }

        return true;
    }
}
