<?php

declare(strict_types=1);

namespace Amasty\FacebookPixelPro\Cron;

use Amasty\FacebookPixelPro\Api\Data\EventQueueInterface;
use Amasty\FacebookPixelPro\Model\QueueService;
use Amasty\FacebookPixelPro\Model\RequestProcessor;
use Magento\Framework\Serialize\SerializerInterface;
use Psr\Log\LoggerInterface;

class DelayedDispatch
{
    public const QUEUE_SIZE = 50;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var RequestProcessor
     */
    private $requestProcessor;

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var QueueService
     */
    private $queueService;

    public function __construct(
        LoggerInterface $logger,
        RequestProcessor $requestProcessor,
        SerializerInterface $serializer,
        QueueService $queueService
    ) {
        $this->logger = $logger;
        $this->requestProcessor = $requestProcessor;
        $this->serializer = $serializer;
        $this->queueService = $queueService;
    }

    /**
     * @return void
     */
    public function execute(): void
    {
        while (!$this->queueService->isQueueEmpty()) {
            try {
                $batch = $this->queueService->shuffleQueues(self::QUEUE_SIZE);
                $groupedQueue = $this->groupByPixelId($batch);
                foreach ($groupedQueue as $pixelId => $pixelData) {
                    $this->requestProcessor->sendRequest(
                        (string)$pixelId,
                        $pixelData['token'],
                        $pixelData['events_data'],
                        $pixelData['test_event_code']
                    );
                }
            } catch (\Exception $e) {
                $this->logger->critical($e);
            }
        }
    }

    /**
     * @param array $items
     * @return array
     */
    private function groupByPixelId(array $items): array
    {
        $result = [];
        /** @var EventQueueInterface $item */
        foreach ($items as $item) {
            if (!array_key_exists($item->getPixelId(), $result)) {
                $result[$item->getPixelId()] = [
                    'token' => $item->getToken(),
                    'test_event_code' => $item->getTestEventCode(),
                    'events_data' => []
                ];
            }

            $result[$item->getPixelId()]['events_data'][] = $this->serializer->unserialize($item->getEventData());
        }

        return $result;
    }
}
