<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Facebook Pixel Base for Magento 2
 */

namespace Amasty\FacebookPixel\ViewModel;

use Amasty\FacebookPixel\Model\ConfigProvider;
use Amasty\FacebookPixel\Model\EventData\EventDataGeneratorPool;
use Amasty\FacebookPixel\Model\PageViewEventResolver;
use Amasty\FacebookPixel\Model\PixelProvider;
use Amasty\FacebookPixel\Model\ThirdParty\ModuleChecker;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Serialize\SerializerInterface;
use Magento\Framework\UrlInterface;
use Magento\Framework\View\Element\Block\ArgumentInterface;
use Psr\Log\LoggerInterface;

class FacebookPixel implements ArgumentInterface
{
    /**
     * @var PageViewEventResolver
     */
    private $pageViewEventResolver;

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var UrlInterface
     */
    private $url;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var PixelProvider
     */
    private $pixelProvider;

    /**
     * @var EventDataGeneratorPool
     */
    private $eventDataGeneratorPool;

    /**
     * @var ModuleChecker
     */
    private $moduleChecker;

    public function __construct(
        PageViewEventResolver $pageViewEventResolver,
        ConfigProvider $configProvider,
        SerializerInterface $serializer,
        UrlInterface $url,
        PixelProvider $pixelProvider,
        EventDataGeneratorPool $eventDataGeneratorPool,
        LoggerInterface $logger,
        ModuleChecker $moduleChecker
    ) {
        $this->pageViewEventResolver = $pageViewEventResolver;
        $this->logger = $logger;
        $this->configProvider = $configProvider;
        $this->serializer = $serializer;
        $this->url = $url;
        $this->pixelProvider = $pixelProvider;
        $this->eventDataGeneratorPool = $eventDataGeneratorPool;
        $this->moduleChecker = $moduleChecker;
    }

    /**
     * @return string|null
     */
    public function getEventDataJson(): ?string
    {
        $eventCode = $this->getEventCode();
        if (!$eventCode) {
            return null;
        }

        try {
            $eventDataGenerator = $this->eventDataGeneratorPool->getDataGenerator($eventCode);
            $eventData = $eventDataGenerator->getEventData();
        } catch (LocalizedException $e) {
            return null;
        } catch (\Exception $e) {
            $this->logger->critical($e);
            return null;
        }

        return $this->serializer->serialize($eventData);
    }

    /**
     * @return string|null
     */
    public function getEventCode(): ?string
    {
        return $this->pageViewEventResolver->getEventKey();
    }

    /**
     * @return string
     */
    public function getProcessPageViewEventUrl(): string
    {
        return $this->url->getUrl('amasty_fbpixel/pageView/processPageView');
    }

    /**
     * @return bool
     */
    public function isFaceBookPixelEnabled(): bool
    {
        return $this->configProvider->isFacebookPixelEnabled();
    }

    /**
     * @return bool
     */
    public function isAmastyGdprCookieEnabled(): bool
    {
        return $this->moduleChecker->isAmastyGdprCookieEnabled();
    }

    /**
     * @return array
     */
    public function getEnabledPixelsForCurrentStore(): array
    {
        return $this->pixelProvider->getEnabledPixelsForCurrentStore();
    }

    /**
     * @return bool
     */
    public function isEventLoggingEnabled(): bool
    {
        return $this->configProvider->isLoggingEnabled();
    }

    /**
     * @return string
     */
    public function getLoggingUrl(): string
    {
        return $this->url->getUrl('amasty_fbpixel/logEvent/log');
    }
}
