<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Facebook Pixel Base for Magento 2
 */

namespace Amasty\FacebookPixel\Setup\Patch\Data;

use Amasty\FacebookPixel\Api\Data\PixelInterfaceFactory;
use Amasty\FacebookPixel\Api\PixelRepositoryInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Setup\Patch\DataPatchInterface;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\StoreManagerInterface;

class MigrateSettings implements DataPatchInterface
{
    /**
     * @var string[]
     */
    private $settingsMap = [
        'category_view' => 'is_category_view_enabled',
        'product_view' => 'is_product_view_enabled',
        'init_checkout' => 'is_initiate_checkout_enabled',
        'purchase' => 'is_purchase_enabled',
        'product_search' => 'is_search_enabled',
        'add_to_cart' => 'is_add_to_cart_enabled',
        'add_to_wishlist' => 'is_add_to_wishlist_enabled',
        'customer_registration' => 'is_registration_enabled',
    ];

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var PixelRepositoryInterface
     */
    private $pixelRepository;

    /**
     * @var ScopeConfigInterface
     */
    private $scopeConfig;

    /**
     * @var PixelInterfaceFactory
     */
    private $pixelFactory;

    public function __construct(
        StoreManagerInterface $storeManager,
        PixelRepositoryInterface $pixelRepository,
        ScopeConfigInterface $scopeConfig,
        PixelInterfaceFactory $pixelFactory
    ) {
        $this->storeManager = $storeManager;
        $this->pixelRepository = $pixelRepository;
        $this->scopeConfig = $scopeConfig;
        $this->pixelFactory = $pixelFactory;
    }

    /**
     * @return $this
     */
    public function apply(): self
    {
        $pixelsData = [];
        $duplicates = [];
        foreach ($this->storeManager->getStores() as $store) {
            if ($this->isPixelEnabled((int)$store->getId())) {
                $storeData = $this->getStoreData((int)$store->getId());
                $isDuplicate = false;
                foreach ($pixelsData as $storeId => $pixelData) {
                    if (empty(array_diff($pixelData, $storeData))) {
                        $duplicates[$storeId][] = $store->getId();
                        $isDuplicate = true;
                    }
                }

                if (!$isDuplicate) {
                    $pixelsData[$store->getId()] = $this->getStoreData((int)$store->getId());
                }
            }
        }

        foreach ($pixelsData as $storeId => &$pixelData) {
            $stores = [$storeId];
            if (isset($duplicates[$storeId])) {
                // phpcs:ignore Magento2.Performance.ForeachArrayMerge.ForeachArrayMerge
                $stores = array_merge($stores, $duplicates[$storeId]);
            }

            $pixelData['stores'] = $stores;
        }

        $this->generatePixels($pixelsData);

        return $this;
    }

    /**
     * @param array $pixelsData
     * @return void
     */
    private function generatePixels(array $pixelsData): void
    {
        foreach ($pixelsData as $pixelData) {
            $model = $this->pixelFactory->create();
            $model->setData($pixelData);
            $this->pixelRepository->save($model);
        }
    }

    /**
     * @param int $storeId
     * @return array
     */
    private function getStoreData(int $storeId): array
    {
        $pixelId = $this->getPixelId($storeId);
        if (!$pixelId) {
            return [];
        }

        $result = [
            'pixel_id' => $pixelId,
            'is_active' => 1
        ];
        foreach ($this->settingsMap as $configSetting => $modelProperty) {
            $result[$modelProperty] = $this->isEventEnabled($configSetting, $storeId);
        }

        return $result;
    }

    /**
     * @param int $storeId
     * @return string|null
     */
    private function getPixelId(int $storeId): ?string
    {
        return $this->scopeConfig->getValue(
            'amasty_facebook_pixel/general/pixel_id',
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * @param int $storeId
     * @return bool
     */
    private function isPixelEnabled(int $storeId): bool
    {
        return $this->scopeConfig->isSetFlag(
            'amasty_facebook_pixel/general/is_enabled',
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * @param string $setting
     * @param int $storeId
     * @return bool
     */
    private function isEventEnabled(string $setting, int $storeId): bool
    {
        return $this->scopeConfig->isSetFlag(
            'amasty_facebook_pixel/events/' . $setting,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * @return array
     */
    public function getAliases(): array
    {
        return [];
    }

    /**
     * @return array
     */
    public static function getDependencies(): array
    {
        return [];
    }
}
