<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Facebook Pixel Base for Magento 2
 */

namespace Amasty\FacebookPixel\Model\ResourceModel;

use Amasty\FacebookPixel\Api\Data\PixelInterface;
use Magento\Framework\DB\Helper;
use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\ResourceModel\Db\AbstractDb;
use Magento\Framework\Model\ResourceModel\Db\Context;

class Pixel extends AbstractDb
{
    /**
     * @var Helper
     */
    private $dbHelper;

    public function __construct(
        Helper $dbHelper,
        Context $context,
        $connectionName = null
    ) {
        parent::__construct($context, $connectionName);
        $this->dbHelper = $dbHelper;
    }

    /**
     * Define resource model
     *
     * @return void
     */
    protected function _construct()
    {
        $this->_init(PixelInterface::MAIN_TABLE, PixelInterface::ENTITY_ID);
    }

    /**
     * @param AbstractModel $object
     */
    protected function _afterSave(AbstractModel $object)
    {
        $this->updateStores($object);

        return parent::_afterSave($object);
    }

    /**
     * @inheritdoc
     */
    protected function _getLoadSelect($field, $value, $object)
    {
        $select = parent::_getLoadSelect($field, $value, $object);

        $table = $this->getMainTable();
        $select->joinLeft(
            ['stores' => $this->getTable(PixelInterface::STORES_TABLE)],
            $table . '.'
            . PixelInterface::ENTITY_ID
            . ' = stores.'
            . PixelInterface::STORE_PIXEL_ID_FIELD,
            []
        );
        $this->dbHelper->addGroupConcatColumn(
            $select,
            'stores',
            'stores.store_id'
        );

        return $select;
    }

    /**
     * @param AbstractModel $object
     */
    private function updateStores(AbstractModel $object): void
    {
        if (!$object->hasData(PixelInterface::STORES)) {
            return;
        }

        $connection = $this->getConnection();
        $pixelId = $object->getEntityId();

        $table = $this->getTable(PixelInterface::STORES_TABLE);
        $select = $connection->select()
            ->from($table, PixelInterface::STORE_ID_FIELD)
            ->where(PixelInterface::STORE_PIXEL_ID_FIELD . ' = ?', $pixelId);
        $oldData = $connection->fetchCol($select);
        $newData = is_array($object->getStores()) ? $object->getStores() : explode(',', $object->getStores());

        if (is_array($newData)) {
            $toDelete = array_diff($oldData, $newData);
            $toInsert = array_diff($newData, $oldData);
            $toInsert = array_diff($toInsert, ['']);
        } else {
            $toDelete = $oldData;
            $toInsert = null;
        }

        if (!empty($toDelete)) {
            $deleteSelect = clone $select;
            $deleteSelect->where('store_id IN (?)', $toDelete);
            $query = $connection->deleteFromSelect($deleteSelect, $table);
            $connection->query($query);
        }

        if (!empty($toInsert)) {
            $insertArray = [];
            foreach ($toInsert as $value) {
                $insertArray[] = [
                    PixelInterface::STORE_PIXEL_ID_FIELD => $pixelId,
                    PixelInterface::STORE_ID_FIELD => $value
                ];
            }

            $connection->insertMultiple($table, $insertArray);
        }
    }
}
