<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Facebook Pixel Base for Magento 2
 */

namespace Amasty\FacebookPixel\Model;

use Amasty\FacebookPixel\Api\Data\PixelInterface;
use Amasty\FacebookPixel\Api\Data\PixelInterfaceFactory;
use Amasty\FacebookPixel\Api\PixelRepositoryInterface;
use Amasty\FacebookPixel\Model\ResourceModel\Pixel as PixelResource;
use Amasty\FacebookPixel\Model\ResourceModel\Pixel\CollectionFactory as PixelCollectionFactory;
use Exception;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Api\SearchResultsInterface;
use Magento\Framework\Api\SearchResultsInterfaceFactory;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;

class PixelRepository implements PixelRepositoryInterface
{
    /**
     * @var PixelResource
     */
    private $resource;

    /**
     * @var PixelInterfaceFactory
     */
    private $modelFactory;

    /**
     * @var PixelCollectionFactory
     */
    private $collectionFactory;

    /**
     * @var CollectionProcessorInterface
     */
    private $collectionProcessor;

    /**
     * @var SearchResultsInterfaceFactory
     */
    private $searchResultsFactory;

    public function __construct(
        PixelResource $resource,
        PixelInterfaceFactory $modelFactory,
        PixelCollectionFactory $collectionFactory,
        CollectionProcessorInterface $collectionProcessor,
        SearchResultsInterfaceFactory $searchResultsFactory
    ) {
        $this->resource = $resource;
        $this->modelFactory = $modelFactory;
        $this->collectionFactory = $collectionFactory;
        $this->collectionProcessor = $collectionProcessor;
        $this->searchResultsFactory = $searchResultsFactory;
    }

    /**
     * @param PixelInterface $pixel
     * @return PixelInterface
     * @throws CouldNotSaveException
     */
    public function save(PixelInterface $pixel): PixelInterface
    {
        try {
            $this->resource->save($pixel);
        } catch (Exception $exception) {
            throw new CouldNotSaveException(__($exception->getMessage()));
        }

        return $pixel;
    }

    /**
     * @param int $entityId
     * @return PixelInterface
     * @throws NoSuchEntityException
     */
    public function getById(int $entityId): PixelInterface
    {
        $pixel = $this->modelFactory->create();
        $this->resource->load($pixel, $entityId);

        if (!$pixel->getPixelId()) {
            throw new NoSuchEntityException(__('Pixel with id "%1" does not exist.', $entityId));
        }

        return $pixel;
    }

    /**
     * @param int $entityId
     * @return PixelInterface|null
     */
    public function findById(int $entityId): ?PixelInterface
    {
        $pixel = $this->modelFactory->create();
        $this->resource->load($pixel, $entityId);

        if (!$pixel->getPixelId()) {
            return null;
        }

        return $pixel;
    }

    /**
     * @param SearchCriteriaInterface $searchCriteria
     * @return SearchResultsInterface
     */
    public function getList(SearchCriteriaInterface $searchCriteria): SearchResultsInterface
    {
        $collection = $this->collectionFactory->create();

        $this->collectionProcessor->process($searchCriteria, $collection);
        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($searchCriteria);
        $searchResults->setItems($collection->getItems());
        $searchResults->setTotalCount($collection->getSize());

        return $searchResults;
    }

    /**
     * @param PixelInterface $pixel
     * @return bool
     * @throws CouldNotDeleteException
     */
    public function delete(PixelInterface $pixel): bool
    {
        try {
            $this->resource->delete($pixel);
        } catch (Exception $exception) {
            throw new CouldNotDeleteException(__($exception->getMessage()));
        }

        return true;
    }

    /**
     * @param int $entityId
     * @return bool
     * @throws CouldNotDeleteException
     * @throws NoSuchEntityException
     */
    public function deleteById(int $entityId): bool
    {
        return $this->delete($this->getById($entityId));
    }
}
