<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Facebook Pixel Base for Magento 2
 */

namespace Amasty\FacebookPixel\Model;

use Amasty\FacebookPixel\Model\EventData\EventDataGeneratorPool;
use Magento\Framework\Exception\LocalizedException;
use Psr\Log\LoggerInterface;

class EventProcessor
{
    /**
     * @var EventDataGeneratorPool
     */
    private $eventDataGeneratorPool;

    /**
     * @var EventSession
     */
    private $eventSession;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var PixelProvider
     */
    private $pixelProvider;

    /**
     * @var EventIdProvider
     */
    private $eventIdProvider;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    public function __construct(
        EventDataGeneratorPool $eventDataGeneratorPool,
        ConfigProvider $configProvider,
        EventSession $eventSession,
        PixelProvider $pixelProvider,
        EventIdProvider $eventIdProvider,
        LoggerInterface $logger
    ) {
        $this->eventDataGeneratorPool = $eventDataGeneratorPool;
        $this->eventSession = $eventSession;
        $this->logger = $logger;
        $this->pixelProvider = $pixelProvider;
        $this->eventIdProvider = $eventIdProvider;
        $this->configProvider = $configProvider;
    }

    /**
     * @param string $eventCode
     * @param null $data
     * @param null $eventData
     * @throws LocalizedException
     */
    public function processEvent(string $eventCode, $data = null, $eventData = null): void
    {
        if (!$this->configProvider->isFacebookPixelEnabled()) {
            return;
        }

        $newEventId = $this->eventIdProvider->getNewEventId();
        $eventsData = $this->getEventsDataForPixels($eventCode, $newEventId, $data, $eventData);

        if ($eventsData !== null) {
            $this->eventSession->setEvents($eventsData);
        }
    }

    /**
     * @param string $eventCode
     * @param int $eventId
     * @param null $data
     * @param null $eventData
     * @return array|null
     * @throws LocalizedException
     */
    public function getEventsDataForPixels(string $eventCode, int $eventId, $data = null, $eventData = null): ?array
    {
        $eventsData = [];
        if (!in_array($eventCode, $this->eventDataGeneratorPool->getEventCodes())) {
            return null;
        }

        $eventDataGenerator = $this->eventDataGeneratorPool->getDataGenerator($eventCode);
        $pixels = $this->pixelProvider->getEnabledPixelsForCurrentStore();
        $mainEventData = null;

        foreach ($pixels as $pixel) {
            if (!$eventDataGenerator->isEventEnabledForPixel($pixel)) {
                continue;
            }

            if ($mainEventData === null) {
                try {
                    $mainEventData = [
                        'event_action' => $eventDataGenerator->getEventAction(),
                        'event_type' => $eventDataGenerator->getEventType(),
                        'event_data' => $eventData ?? $eventDataGenerator->getEventData($data),
                        'event_id' => $eventId
                    ];
                } catch (\Exception $e) {
                    $this->logger->critical($e);

                    return $eventsData;
                }
            }

            $eventsData[] = [$pixel->getPixelId() => $mainEventData];
        }

        return $eventsData;
    }
}
