<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Facebook Pixel Base for Magento 2
 */

namespace Amasty\FacebookPixel\Model\EventData\EventDataGenerator;

use Amasty\FacebookPixel\Api\Data\PixelInterface;
use Magento\Bundle\Model\Product\Type;
use Magento\Catalog\Api\Data\CategoryInterface;
use Magento\Checkout\Model\Session as CheckoutSession;
use Magento\Sales\Model\Order;

class Purchase implements EventDataGeneratorInterface
{
    private const EVENT_ACTION = 'trackSingle';
    private const EVENT_TYPE = 'Purchase';

    /**
     * @var CheckoutSession
     */
    private $checkoutSession;

    public function __construct(
        CheckoutSession $checkoutSession
    ) {
        $this->checkoutSession = $checkoutSession;
        $this->checkoutSession = $checkoutSession;
    }

    /**
     * @param null|Order $order
     * @return array|null
     */
    public function getEventData($order = null): ?array
    {
        $order = $order ?? $this->getOrder();
        if ($order === null) {
            return null;
        }

        $orderItems = $order->getAllVisibleItems();
        $itemContent = [];
        $itemCount = 0;
        foreach ($orderItems as $item) {
            $itemContent['contents'][] = [
                'id' => $item->getProduct()->getTypeId() == Type::TYPE_CODE
                    ? $item->getProduct()->getData('sku') : $item->getSku(),
                'quantity' => (int)$item->getQtyOrdered(),
                'item_price' => $item->getPrice()
            ];

            if (empty($itemContent['content_ids'])) {
                $itemContent['content_ids'] = [];
            }

            $itemContent['content_ids'][] = $item->getProduct()->getTypeId() == Type::TYPE_CODE
                ? $item->getProduct()->getData('sku') : $item->getSku();
            $itemCount += $item->getQtyOrdered();
        }

        if ($itemContent) {
            return [
                'content_ids' => $itemContent['content_ids'],
                'contents' => $itemContent['contents'],
                'content_type' => 'product',
                'num_items' => $itemCount,
                'value' => $order->getGrandTotal(),
                'currency' => $order->getOrderCurrencyCode(),
            ];
        }

        return null;
    }

    /**
     * @return string
     */
    public function getEventAction(): string
    {
        return self::EVENT_ACTION;
    }

    /**
     * @return string
     */
    public function getEventType(): string
    {
        return self::EVENT_TYPE;
    }

    /**
     * @param PixelInterface $pixel
     * @return bool
     */
    public function isEventEnabledForPixel(PixelInterface $pixel): bool
    {
        return $pixel->getIsPurchaseEnabled();
    }

    /**
     * @return CategoryInterface
     */
    private function getOrder(): Order
    {
        return $this->checkoutSession->getLastRealOrder();
    }
}
