<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Facebook Pixel Base for Magento 2
 */

namespace Amasty\FacebookPixel\Model\EventData\EventDataGenerator;

use Amasty\FacebookPixel\Api\Data\PixelInterface;
use Magento\Catalog\Api\Data\CategoryInterface;
use Magento\Catalog\Api\Data\ProductInterface;
use Magento\Framework\Registry;
use Magento\Store\Api\Data\StoreInterface;

class ProductView implements EventDataGeneratorInterface
{
    private const EVENT_ACTION = 'trackSingle';
    private const EVENT_TYPE = 'ViewContent';

    /**
     * @var Registry
     */
    private $registry;

    /**
     * @var StoreInterface
     */
    private $store;

    public function __construct(
        Registry $registry,
        StoreInterface $store
    ) {
        $this->registry = $registry;
        $this->store = $store;
    }

    /**
     * @param null|ProductInterface $product
     * @return array|null
     */
    public function getEventData($product = null): ?array
    {
        $product = $product ?? $this->getCurrentProduct();
        if ($product === null) {
            return null;
        }

        $eventData = [
            'content_name' => $product->getName(),
            'content_ids' => [$product->getSku()],
            'content_type' => 'product',
            'value' => $product->getFinalPrice(),
            'currency' => $this->store->getCurrentCurrencyCode()
        ];

        if ($category = $this->getCurrentCategory()) {
            $eventData['content_category'] = $category->getName();
        }

        return $eventData;
    }

    /**
     * @return string
     */
    public function getEventAction(): string
    {
        return self::EVENT_ACTION;
    }

    /**
     * @return string
     */
    public function getEventType(): string
    {
        return self::EVENT_TYPE;
    }

    /**
     * @param PixelInterface $pixel
     * @return bool
     */
    public function isEventEnabledForPixel(PixelInterface $pixel): bool
    {
        return $pixel->getIsProductViewEnabled();
    }

    /**
     * @return ProductInterface|null
     */
    private function getCurrentProduct(): ?ProductInterface
    {
        return $this->registry->registry('current_product');
    }

    /**
     * @return CategoryInterface|null
     */
    private function getCurrentCategory(): ?CategoryInterface
    {
        return $this->registry->registry('current_category');
    }
}
