<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Facebook Pixel Base for Magento 2
 */

namespace Amasty\FacebookPixel\Controller\Adminhtml\Pixels;

use Amasty\FacebookPixel\Api\Data\PixelInterface;
use Amasty\FacebookPixel\Api\Data\PixelInterfaceFactory;
use Amasty\FacebookPixel\Api\PixelRepositoryInterface;
use Magento\Backend\App\Action;
use Magento\Backend\App\Action\Context;
use Magento\Backend\Model\View\Result\Redirect;
use Magento\Framework\Api\DataObjectHelper;
use Magento\Framework\Controller\ResultInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Psr\Log\LoggerInterface;

class Save extends Action
{
    const ADMIN_RESOURCE = 'Amasty_FacebookPixel::manage_pixels';

    /**
     * @var PixelRepositoryInterface
     */
    private $pixelRepository;

    /**
     * @var DataObjectHelper
     */
    private $dataObjectHelper;

    /**
     * @var PixelInterfaceFactory
     */
    private $pixelFactory;

    /**
     * @var LoggerInterface
     */
    private $logger;

    public function __construct(
        Context $context,
        PixelRepositoryInterface $pixelRepository,
        DataObjectHelper $dataObjectHelper,
        PixelInterfaceFactory $pixelFactory,
        LoggerInterface $logger
    ) {
        parent::__construct($context);
        $this->pixelRepository = $pixelRepository;
        $this->dataObjectHelper = $dataObjectHelper;
        $this->pixelFactory = $pixelFactory;
        $this->logger = $logger;
    }

    /**
     * @return ResultInterface
     */
    public function execute(): ResultInterface
    {
        /** @var Redirect $resultRedirect */
        $resultRedirect = $this->resultRedirectFactory->create();
        $data = $this->getRequest()->getPostValue();
        $this->prepareData($data);

        $id = $this->getRequest()->getParam(PixelInterface::ENTITY_ID);

        try {
            if ($id) {
                $pixel = $this->pixelRepository->getById((int)$id);
            } else {
                $pixel = $this->pixelFactory->create();
            }
        } catch (NoSuchEntityException $e) {
            $pixel = $this->pixelFactory->create();
        }

        $this->dataObjectHelper->populateWithArray(
            $pixel,
            $data,
            PixelInterface::class
        );

        try {
            $this->pixelRepository->save($pixel);
            $this->messageManager->addSuccessMessage(__('Pixel have been saved.'));

            if ($this->getRequest()->getParam('back')) {
                return $resultRedirect->setPath('*/*/edit', ['id' => $pixel->getId()]);
            }
        } catch (LocalizedException $e) {
            $this->messageManager->addErrorMessage($e->getMessage());
        } catch (\Exception $e) {
            $this->messageManager->addExceptionMessage($e, __('Something went wrong while saving Pixel.'));
            $this->logger->critical($e);
        }

        return $resultRedirect->setPath(
            '*/*/',
            ['id' => $pixel->getId() ?? $this->getRequest()->getParam('id')]
        );
    }

    /**
     * @param array $data
     * @return void
     */
    public function prepareData(array &$data): void
    {
        if (isset($data['stores'])) {
            $data['stores'] = implode(',', $data['stores']);
        }
    }
}
